/**
 * statistics.js - وظائف عرض الإحصائيات والرسومات البيانية
 */

// المتغيرات العامة
let usersChart;
let searchesChart;
let osChart;

// تحميل الإحصائيات عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', () => {
    // التحقق من صلاحيات المستخدم
    if (!auth.isAdmin()) {
        window.location.href = 'index.html';
        return;
    }
    
    // تحديث واجهة المستخدم بناءً على حالة تسجيل الدخول
    const currentUser = auth.getCurrentUser();
    if (currentUser) {
        updateUIForAuthStatus(currentUser);
    }
    
    // تحميل الإحصائيات
    loadStatistics();
    
    // إضافة مستمع الحدث لزر تسجيل الخروج
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', () => {
            auth.handleLogout();
        });
    }
});

/**
 * تحميل وعرض الإحصائيات
 */
function loadStatistics() {
    // الحصول على بيانات المستخدمين
    const users = auth.getUsers();
    
    // حساب الإحصائيات العامة
    const totalUsers = users.length;
    const activeUsers = users.filter(user => user.isActive === true).length;
    const blockedUsers = users.filter(user => user.isActive === false).length;
    const pendingUsers = users.filter(user => user.activationPending === true).length;
    
    // حساب إجمالي عمليات البحث
    let totalSearches = 0;
    let allSearches = [];
    users.forEach(user => {
        const userSearches = user.savedSearches || [];
        totalSearches += userSearches.length;
        
        // جمع كل عمليات البحث مع معلومات المستخدم
        userSearches.forEach(search => {
            allSearches.push({
                ...search,
                userName: user.name,
                userEmail: user.email,
                userId: user.id
            });
        });
    });
    
    // حساب متوسط النقاط لكل مستخدم
    const totalPoints = users.reduce((sum, user) => sum + (user.points || 0), 0);
    const averagePoints = totalUsers > 0 ? Math.round(totalPoints / totalUsers) : 0;
    
    // تحديث الإحصائيات العامة في الصفحة
    document.getElementById('totalUsersCount').textContent = totalUsers;
    document.getElementById('totalSearchesCount').textContent = totalSearches;
    document.getElementById('averagePointsCount').textContent = averagePoints;
    document.getElementById('activeUsersCount').textContent = activeUsers;
    document.getElementById('blockedUsersCount').textContent = blockedUsers;
    
    // إنشاء الرسومات البيانية
    createUsersChart(activeUsers, blockedUsers, pendingUsers);
    createSearchesChart(allSearches);
    createOsChart(users);
    
    // عرض الكلمات المفتاحية الأكثر بحثًا
    displayTopKeywords(allSearches);
}

/**
 * إنشاء رسم بياني لتوزيع المستخدمين
 * @param {number} activeUsers - عدد المستخدمين النشطين
 * @param {number} blockedUsers - عدد المستخدمين المحظورين
 * @param {number} pendingUsers - عدد المستخدمين الذين ينتظرون التفعيل
 */
function createUsersChart(activeUsers, blockedUsers, pendingUsers) {
    const ctx = document.getElementById('usersChart').getContext('2d');
    
    // تدمير الرسم البياني السابق إذا كان موجودًا
    if (usersChart) {
        usersChart.destroy();
    }
    
    // إنشاء رسم بياني جديد
    usersChart = new Chart(ctx, {
        type: 'pie',
        data: {
            labels: ['مستخدمون نشطون', 'مستخدمون محظورون', 'حسابات تنتظر التفعيل'],
            datasets: [{
                data: [activeUsers, blockedUsers, pendingUsers],
                backgroundColor: [
                    'rgba(40, 167, 69, 0.7)',  // أخضر للمستخدمين النشطين
                    'rgba(220, 53, 69, 0.7)',   // أحمر للمستخدمين المحظورين
                    'rgba(255, 193, 7, 0.7)'    // أصفر للحسابات التي تنتظر التفعيل
                ],
                borderColor: [
                    'rgba(40, 167, 69, 1)',
                    'rgba(220, 53, 69, 1)',
                    'rgba(255, 193, 7, 1)'
                ],
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom',
                },
                title: {
                    display: true,
                    text: 'توزيع المستخدمين حسب الحالة'
                }
            }
        }
    });
}

/**
 * إنشاء رسم بياني لعمليات البحث
 * @param {Array} searches - قائمة عمليات البحث
 */
function createSearchesChart(searches) {
    // ترتيب عمليات البحث حسب التاريخ
    searches.sort((a, b) => new Date(b.date) - new Date(a.date));
    
    // أخذ آخر 7 عمليات بحث
    const recentSearches = searches.slice(0, 7);
    
    // تحضير البيانات للرسم البياني
    const labels = recentSearches.map(search => {
        const date = new Date(search.date);
        return date.toLocaleDateString('ar-SA');
    });
    
    const data = recentSearches.map(search => search.resultCount || 0);
    
    const ctx = document.getElementById('searchesChart').getContext('2d');
    
    // تدمير الرسم البياني السابق إذا كان موجودًا
    if (searchesChart) {
        searchesChart.destroy();
    }
    
    // إنشاء رسم بياني جديد
    searchesChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'عدد النتائج',
                data: data,
                backgroundColor: 'rgba(13, 110, 253, 0.7)',
                borderColor: 'rgba(13, 110, 253, 1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'عدد النتائج'
                    }
                },
                x: {
                    title: {
                        display: true,
                        text: 'التاريخ'
                    }
                }
            },
            plugins: {
                title: {
                    display: true,
                    text: 'عمليات البحث الأخيرة'
                }
            }
        }
    });
}

/**
 * إنشاء رسم بياني لأنظمة التشغيل
 * @param {Array} users - قائمة المستخدمين
 */
function createOsChart(users) {
    // جمع إحصائيات أنظمة التشغيل
    const osStats = {};
    
    users.forEach(user => {
        const os = user.deviceInfo?.os || 'غير معروف';
        osStats[os] = (osStats[os] || 0) + 1;
    });
    
    // تحويل البيانات إلى تنسيق مناسب للرسم البياني
    const labels = Object.keys(osStats);
    const data = Object.values(osStats);
    
    // إنشاء ألوان عشوائية لكل نظام تشغيل
    const backgroundColors = labels.map(() => {
        const r = Math.floor(Math.random() * 200);
        const g = Math.floor(Math.random() * 200);
        const b = Math.floor(Math.random() * 200);
        return `rgba(${r}, ${g}, ${b}, 0.7)`;
    });
    
    const borderColors = backgroundColors.map(color => color.replace('0.7', '1'));
    
    const ctx = document.getElementById('osChart').getContext('2d');
    
    // تدمير الرسم البياني السابق إذا كان موجودًا
    if (osChart) {
        osChart.destroy();
    }
    
    // إنشاء رسم بياني جديد
    osChart = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: labels,
            datasets: [{
                data: data,
                backgroundColor: backgroundColors,
                borderColor: borderColors,
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom',
                },
                title: {
                    display: true,
                    text: 'توزيع أنظمة التشغيل'
                }
            }
        }
    });
}

/**
 * عرض الكلمات المفتاحية الأكثر بحثًا
 * @param {Array} searches - قائمة عمليات البحث
 */
function displayTopKeywords(searches) {
    // جمع إحصائيات الكلمات المفتاحية
    const keywordStats = {};
    
    searches.forEach(search => {
        const keyword = search.keyword || '';
        if (keyword) {
            keywordStats[keyword] = (keywordStats[keyword] || 0) + 1;
        }
    });
    
    // تحويل البيانات إلى مصفوفة وترتيبها تنازليًا
    const keywordArray = Object.entries(keywordStats).map(([keyword, count]) => ({
        keyword,
        count
    }));
    
    keywordArray.sort((a, b) => b.count - a.count);
    
    // أخذ أعلى 10 كلمات مفتاحية
    const topKeywords = keywordArray.slice(0, 10);
    
    // عرض الكلمات المفتاحية
    const topKeywordsContainer = document.getElementById('topKeywords');
    topKeywordsContainer.innerHTML = '';
    
    topKeywords.forEach(item => {
        const col = document.createElement('div');
        col.className = 'col-md-3 col-sm-6 mb-3';
        
        const card = document.createElement('div');
        card.className = 'card h-100';
        
        const cardBody = document.createElement('div');
        cardBody.className = 'card-body text-center';
        
        const keyword = document.createElement('h5');
        keyword.className = 'card-title';
        keyword.textContent = item.keyword;
        
        const count = document.createElement('p');
        count.className = 'card-text';
        count.innerHTML = `<span class="badge bg-primary">${item.count}</span> مرة`;
        
        cardBody.appendChild(keyword);
        cardBody.appendChild(count);
        card.appendChild(cardBody);
        col.appendChild(card);
        
        topKeywordsContainer.appendChild(col);
    });
    
    // إذا لم تكن هناك كلمات مفتاحية
    if (topKeywords.length === 0) {
        const noKeywords = document.createElement('div');
        noKeywords.className = 'col-12 text-center';
        noKeywords.textContent = 'لا توجد كلمات مفتاحية للعرض';
        topKeywordsContainer.appendChild(noKeywords);
    }
}

/**
 * تحديث واجهة المستخدم بناءً على حالة تسجيل الدخول
 * @param {Object} user - بيانات المستخدم
 */
function updateUIForAuthStatus(user) {
    const userInfo = document.getElementById('userInfo');
    const loginBtn = document.getElementById('loginBtn');
    const userName = document.getElementById('userName');
    const userPoints = document.getElementById('userPoints');
    
    if (user) {
        // إظهار معلومات المستخدم
        userInfo.classList.remove('d-none');
        loginBtn.classList.add('d-none');
        
        // تحديث اسم المستخدم والنقاط
        userName.textContent = user.name;
        userPoints.textContent = `${user.points || 0} نقطة`;
    } else {
        // إخفاء معلومات المستخدم وإظهار زر تسجيل الدخول
        userInfo.classList.add('d-none');
        loginBtn.classList.remove('d-none');
    }
}
