const express = require('express');
const cors = require('cors');
const https = require('https');
const http = require('http');
const path = require('path');
const { URL } = require('url');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 8080;

console.log('🚀 Starting WhatsApp Business Scraper Server...');
console.log('💾 Memory-optimized mode (native modules only)');

// Middleware
app.use(cors());
app.use(express.json({ limit: '1mb' })); // Limit payload size
app.use(express.static(path.join(__dirname), {
    maxAge: '1d', // Cache static files
    etag: true
}));

// ============================================
// HELPER FUNCTIONS
// ============================================

/**
 * Lightweight HTTPS GET request (replaces axios)
 * @param {string} url - Full URL to fetch
 * @returns {Promise<Object>} Parsed JSON response
 */
function httpsGet(url) {
    return new Promise((resolve, reject) => {
        const parsedUrl = new URL(url);
        const protocol = parsedUrl.protocol === 'https:' ? https : http;

        const options = {
            hostname: parsedUrl.hostname,
            port: parsedUrl.port,
            path: parsedUrl.pathname + parsedUrl.search,
            method: 'GET',
            timeout: 15000, // 15s timeout
            headers: {
                'User-Agent': 'WhatsApp-Business-Scraper/1.0',
                'Accept': 'application/json'
            }
        };

        const req = protocol.request(options, (res) => {
            let data = '';

            // Handle redirects
            if (res.statusCode >= 300 && res.statusCode < 400 && res.headers.location) {
                return httpsGet(res.headers.location).then(resolve).catch(reject);
            }

            res.on('data', (chunk) => {
                data += chunk;
                // Prevent memory overflow
                if (data.length > 5 * 1024 * 1024) { // 5MB limit
                    req.abort();
                    reject(new Error('Response too large'));
                }
            });

            res.on('end', () => {
                try {
                    const parsed = JSON.parse(data);
                    resolve(parsed);
                } catch (e) {
                    reject(new Error('Invalid JSON response'));
                }
            });
        });

        req.on('error', reject);
        req.on('timeout', () => {
            req.abort();
            reject(new Error('Request timeout'));
        });

        req.end();
    });
}

/**
 * Build Google Places API URL
 */
function buildPlacesUrl(keyword, location, apiKey) {
    const baseUrl = 'https://maps.googleapis.com/maps/api/place/textsearch/json';
    const query = `${keyword} in ${location}`;
    const params = new URLSearchParams({
        query: query,
        key: apiKey,
        language: 'ar'
    });
    return `${baseUrl}?${params.toString()}`;
}

/**
 * Get Place Details
 */
function buildPlaceDetailsUrl(placeId, apiKey) {
    const baseUrl = 'https://maps.googleapis.com/maps/api/place/details/json';
    const params = new URLSearchParams({
        place_id: placeId,
        fields: 'name,formatted_address,formatted_phone_number,place_id,rating,user_ratings_total,opening_hours',
        key: apiKey,
        language: 'ar'
    });
    return `${baseUrl}?${params.toString()}`;
}

// ============================================
// ROUTES
// ============================================

// Health check endpoint
app.get('/health', (req, res) => {
    res.json({
        status: 'healthy',
        uptime: process.uptime(),
        memory: process.memoryUsage(),
        timestamp: new Date().toISOString()
    });
});

// Serve HTML pages
app.get('/', (req, res) => res.sendFile(path.join(__dirname, 'index.html')));
app.get('/admin-panel.html', (req, res) => res.sendFile(path.join(__dirname, 'admin-panel.html')));
app.get('/login.html', (req, res) => res.sendFile(path.join(__dirname, 'login.html')));

// ============================================
// GOOGLE PLACES API PROXY
// ============================================

app.get('/api/places/search', async (req, res) => {
    const startTime = Date.now();

    try {
        const { keyword, location, resultCount, apiKey: apiKeyOverride } = req.query;

        // Validation
        if (!keyword || !location) {
            return res.status(400).json({
                error: 'Missing required parameters',
                details: 'Both keyword and location are required'
            });
        }

        // Get API key (priority: override > env > hardcoded fallback)
        const apiKey = apiKeyOverride ||
            process.env.GOOGLE_MAPS_API_KEY ||
            'AIzaSyAYGuFwowyd5OZ-ZTkw1MQMqpHdKturxS8';

        const requestedCount = parseInt(resultCount) || 20;
        const maxResults = Math.min(requestedCount, 60); // Google limitation

        console.log(`🔍 Search: "${keyword}" in "${location}" (limit: ${maxResults})`);

        // Fetch search results
        const searchUrl = buildPlacesUrl(keyword, location, apiKey);
        const data = await httpsGet(searchUrl);

        // Check API response status
        if (data.status === 'REQUEST_DENIED') {
            console.error('❌ Google API: Request denied');
            return res.status(403).json({
                error: 'API Request Denied',
                details: data.error_message || 'Invalid API key or permissions',
                status: data.status
            });
        }

        if (data.status === 'OVER_QUERY_LIMIT') {
            console.error('❌ Google API: Quota exceeded');
            return res.status(429).json({
                error: 'API Quota Exceeded',
                details: 'Daily quota limit reached. Try again tomorrow.',
                status: data.status
            });
        }

        if (data.status === 'ZERO_RESULTS') {
            console.log('ℹ️ No results found');
            return res.json({ results: [], count: 0 });
        }

        if (!data.results || data.status !== 'OK') {
            console.error('❌ Google API unexpected status:', data.status);
            return res.status(500).json({
                error: 'API Error',
                details: data.error_message || data.status,
                status: data.status
            });
        }

        // Return basic results (no details fetching to save quota & memory)
        const results = data.results.slice(0, maxResults);

        const duration = Date.now() - startTime;
        console.log(`✅ Found ${results.length} results in ${duration}ms`);

        res.json({
            results: results,
            count: results.length,
            status: 'success',
            duration_ms: duration
        });

    } catch (error) {
        const duration = Date.now() - startTime;
        console.error('❌ Search error:', error.message);

        res.status(500).json({
            error: 'Internal Server Error',
            details: error.message,
            duration_ms: duration
        });
    }
});

// Fetch detailed information for a specific place
app.get('/api/places/details/:placeId', async (req, res) => {
    try {
        const { placeId } = req.params;
        const { apiKey: apiKeyOverride } = req.query;

        if (!placeId) {
            return res.status(400).json({ error: 'Missing placeId' });
        }

        const apiKey = apiKeyOverride ||
            process.env.GOOGLE_MAPS_API_KEY ||
            'AIzaSyAYGuFwowyd5OZ-ZTkw1MQMqpHdKturxS8';

        const detailsUrl = buildPlaceDetailsUrl(placeId, apiKey);
        const data = await httpsGet(detailsUrl);

        if (data.status !== 'OK' || !data.result) {
            return res.status(404).json({
                error: 'Place not found',
                details: data.error_message || data.status
            });
        }

        res.json({
            result: data.result,
            status: 'success'
        });

    } catch (error) {
        console.error('❌ Details error:', error.message);
        res.status(500).json({
            error: 'Internal Server Error',
            details: error.message
        });
    }
});

// ============================================
// WHATSAPP ENDPOINTS (Disabled - Cloud API recommended)
// ============================================

app.post('/api/whatsapp-qr/start', (req, res) => {
    res.status(503).json({
        error: 'WhatsApp QR Service Disabled',
        message: 'Please use WhatsApp Cloud API instead (more reliable and doesn\'t require Chrome)',
        docs: 'https://developers.facebook.com/docs/whatsapp/cloud-api'
    });
});

app.get('/api/whatsapp-qr/status', (req, res) => {
    res.json({
        success: false,
        status: {
            ready: false,
            error: 'QR service disabled. Use Cloud API.'
        }
    });
});

app.post('/api/whatsapp/send', (req, res) => {
    res.status(503).json({
        error: 'Direct messaging disabled',
        message: 'Configure WhatsApp Cloud API via Settings panel'
    });
});

// ============================================
// AI GENERATION ENDPOINT (Future enhancement)
// ============================================

app.post('/api/ai/generate', (req, res) => {
    res.status(501).json({
        error: 'Not Implemented',
        message: 'AI generation will be added in Phase 3'
    });
});

// ============================================
// ERROR HANDLING
// ============================================

// 404 handler
app.use((req, res) => {
    res.status(404).json({
        error: 'Not Found',
        path: req.path
    });
});

// Global error handler
app.use((err, req, res, next) => {
    console.error('💥 Unhandled error:', err);
    res.status(500).json({
        error: 'Internal Server Error',
        message: process.env.NODE_ENV === 'production' ? 'Something went wrong' : err.message
    });
});

// ============================================
// SERVER STARTUP
// ============================================

const server = app.listen(PORT, () => {
    console.log('✅ Server successfully started');
    console.log(`🌐 URL: http://localhost:${PORT}`);
    console.log(`📊 Memory: ${Math.round(process.memoryUsage().heapUsed / 1024 / 1024)}MB`);
    console.log(`⏰ Time: ${new Date().toLocaleString('ar-SA')}`);
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
});

// Graceful shutdown
process.on('SIGTERM', () => {
    console.log('📴 SIGTERM received, shutting down gracefully');
    server.close(() => {
        console.log('👋 Server closed');
        process.exit(0);
    });
});

process.on('SIGINT', () => {
    console.log('📴 SIGINT received, shutting down gracefully');
    server.close(() => {
        console.log('👋 Server closed');
        process.exit(0);
    });
});

// Handle uncaught exceptions
process.on('uncaughtException', (err) => {
    console.error('💥 Uncaught Exception:', err);
    // Don't exit in production, log and continue
    if (process.env.NODE_ENV !== 'production') {
        process.exit(1);
    }
});

process.on('unhandledRejection', (reason, promise) => {
    console.error('💥 Unhandled Rejection at:', promise, 'reason:', reason);
    // Don't exit in production
    if (process.env.NODE_ENV !== 'production') {
        process.exit(1);
    }
});
