"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.load = void 0;
const debug_1 = require("./debug");
const calcCharacterWidth = function (lines) {
    let minLeft = Infinity;
    let maxRight = -Infinity;
    lines.forEach(l => {
        let left = l.length;
        for (let i = 0; i < l.length; i++) {
            if (l[i] !== ' ') {
                left = i;
                break;
            }
        }
        let right = 0;
        for (let i = l.length - 1; i >= 0; i--) {
            if (l[i] !== ' ') {
                right = i;
                break;
            }
        }
        minLeft = Math.min(left, minLeft);
        maxRight = Math.max(right, maxRight);
        debug_1.debugVerbose('minLeft:', minLeft, 'maxRight:', maxRight);
    });
    return maxRight - minLeft + 1;
};
const formatCharacter = function (info, maxCharHeight) {
    const { codes, defs } = info;
    const content = codes.reduce((text, c) => text + String.fromCharCode(c), '');
    const lines = content.split('\n');
    if (lines.length < maxCharHeight) {
        lines.push(...new Array(maxCharHeight - lines.length).fill(''));
    }
    const maxLen = lines.reduce((max, l) => Math.max(max, l.length), 0);
    for (let i = 0; i < lines.length; i++) {
        lines[i] += ' '.repeat(maxLen - lines[i].length);
    }
    const width = calcCharacterWidth(lines);
    debug_1.debugVerbose('width:', width, 'maxCharHeight:', maxCharHeight);
    return defs.map(def => ({
        lines: [...lines],
        width,
        height: maxCharHeight,
        def,
    }));
};
const loadCharacters = function (fontFamily) {
    let fontModule;
    if (fontFamily) {
        fontModule = fontFamily;
    }
    else {
        try {
            fontModule = require('./fonts/default.js');
        }
        catch (e) {
            throw Error(`FONT NOT FOUND (Font Family Default). \n${e}`);
        }
    }
    const fonts = fontModule.fonts;
    const maxCharHeight = fonts.reduce((max, f) => Math.max(f.codes.filter(c => c === 10).length + 1, max), 0);
    return fonts.map(f => formatCharacter(f, maxCharHeight));
};
const load = function (fontFamily) {
    var _a;
    debug_1.debug('loading font', (_a = fontFamily === null || fontFamily === void 0 ? void 0 : fontFamily.name) !== null && _a !== void 0 ? _a : 'default');
    const characters = loadCharacters(fontFamily);
    debug_1.debugVerbose('characters:', characters);
    debug_1.debug('load characters successfully:', characters.map(cur => cur.map(c => c.def).join(',')).join(','));
    const mapping = {};
    characters
        .filter(c => c.length)
        .reduce((prev, c) => [...prev, ...c], [])
        .forEach(c => {
        mapping[c.def] = c;
    });
    return mapping;
};
exports.load = load;
