// ============================================
// FIREBASE AUTHENTICATION SYSTEM v2.0
// Memory-optimized, embedded config, retry logic
// ============================================

// Embedded Firebase Configuration (no external file needed)
const FIREBASE_CONFIG = {
    apiKey: "AIzaSyBj2yQAV4QDmTGwtpaBOnYRsrww7csi6_M",
    authDomain: "vcprocode.firebaseapp.com",
    projectId: "vcprocode",
    storageBucket: "vcprocode.firebasestorage.app",
    messagingSenderId: "847443321496",
    appId: "1:847443321496:web:0646818153935bd9a666ed",
    measurementId: "G-PWNMFYZH2E"
};

// Admin email (permanent admin access)
const ADMIN_EMAIL = 'raiedprof@gmail.com';

// Initialize Firebase with retry mechanism
let fireAuth = null;
let db = null;
let initAttempts = 0;
const MAX_INIT_ATTEMPTS = 50;

function initializeFirebase() {
    return new Promise((resolve, reject) => {
        const attemptInit = () => {
            // Check if Firebase SDK is loaded
            if (typeof firebase === 'undefined') {
                initAttempts++;
                if (initAttempts >= MAX_INIT_ATTEMPTS) {
                    reject(new Error('Firebase SDK failed to load after 5 seconds'));
                    return;
                }
                console.log(`⏳ Waiting for Firebase SDK... (attempt ${initAttempts}/${MAX_INIT_ATTEMPTS})`);
                setTimeout(attemptInit, 100);
                return;
            }

            try {
                // Initialize Firebase app
                if (!firebase.apps.length) {
                    firebase.initializeApp(FIREBASE_CONFIG);
                    console.log('✅ Firebase app initialized');
                }

                // Get Auth and Firestore instances
                fireAuth = firebase.auth();
                db = firebase.firestore();

                // Expose globally for compatibility
                window.fireAuth = fireAuth;
                window.db = db;

                console.log('✅ Firebase authentication ready');
                resolve({ fireAuth, db });
            } catch (error) {
                console.error('❌ Firebase initialization error:', error);
                reject(error);
            }
        };

        attemptInit();
    });
}

// ============================================
// AUTHENTICATION STATE MANAGER
// ============================================

const auth = {
    currentUser: null,
    initialized: false,

    /**
     * Initialize authentication system
     */
    async init() {
        try {
            await initializeFirebase();

            // Listen for auth state changes
            fireAuth.onAuthStateChanged(async (user) => {
                if (user) {
                    console.log('👤 User logged in:', user.email);
                    await this.loadUserData(user);
                } else {
                    console.log('👤 No user logged in');
                    this.currentUser = null;
                }
                this.initialized = true;
            });

            return true;
        } catch (error) {
            console.error('❌ Auth init failed:', error);
            this.showError('فشل تحميل نظام المصادقة. الرجاء تحديث الصفحة.');
            return false;
        }
    },

    /**
     * Load user data from Firestore
     */
    async loadUserData(user) {
        try {
            const userDoc = await db.collection('users').doc(user.uid).get();

            if (userDoc.exists) {
                this.currentUser = {
                    id: user.uid,
                    email: user.email,
                    ...userDoc.data()
                };

                // Force admin status for admin email
                if (user.email === ADMIN_EMAIL) {
                    this.currentUser.isAdmin = true;
                    this.currentUser.points = Infinity;

                    // Update Firestore if needed
                    if (!userDoc.data().isAdmin) {
                        await db.collection('users').doc(user.uid).update({
                            isAdmin: true,
                            points: 999999999
                        });
                    }
                }
            } else {
                // Create new user document
                this.currentUser = {
                    id: user.uid,
                    email: user.email,
                    displayName: user.displayName || 'مستخدم جديد',
                    points: 100,
                    isAdmin: user.email === ADMIN_EMAIL,
                    createdAt: new Date().toISOString(),
                    lastLogin: new Date().toISOString()
                };

                await db.collection('users').doc(user.uid).set(this.currentUser);
                console.log('✅ New user profile created');
            }

            // Cache in localStorage
            this.cacheUser(this.currentUser);

        } catch (error) {
            console.error('❌ Error loading user data:', error);
            this.showError('خطأ في تحميل بيانات المستخدم');
        }
    },

    /**
     * Get current user (with cache)
     */
    getCurrentUser() {
        if (this.currentUser) {
            return this.currentUser;
        }

        // Try to load from cache
        try {
            const cached = localStorage.getItem('currentUser');
            if (cached) {
                this.currentUser = JSON.parse(cached);
                return this.currentUser;
            }
        } catch (e) {
            console.warn('Failed to load cached user');
        }

        return null;
    },

    /**
     * Cache user data
     */
    cacheUser(user) {
        try {
            localStorage.setItem('currentUser', JSON.stringify(user));
        } catch (e) {
            console.warn('Failed to cache user');
        }
    },

    /**
     * Check if current user is admin
     */
    isAdmin() {
        const user = this.getCurrentUser();
        return user && (user.isAdmin === true || user.email === ADMIN_EMAIL);
    },

    /**
     * Get user points (or Infinity for admin)
     */
    getPoints() {
        const user = this.getCurrentUser();
        if (!user) return 0;
        if (this.isAdmin()) return Infinity;
        return user.points || 0;
    },

    /**
     * Deduct points from user
     */
    async deductPoints(amount) {
        const user = this.getCurrentUser();
        if (!user) {
            throw new Error('User not logged in');
        }

        // Skip deduction for admin
        if (this.isAdmin()) {
            console.log('💎 Admin: Points deduction skipped');
            return true;
        }

        // Check if user has enough points
        if (user.points < amount) {
            throw new Error('Insufficient points');
        }

        try {
            const newPoints = user.points - amount;

            // Update Firestore
            await db.collection('users').doc(user.id).update({
                points: newPoints,
                lastActivity: new Date().toISOString()
            });

            // Update local copy
            this.currentUser.points = newPoints;
            this.cacheUser(this.currentUser);

            console.log(`✅ Points deducted: ${amount} (remaining: ${newPoints})`);
            return true;

        } catch (error) {
            console.error('❌ Error deducting points:', error);
            throw new Error('Failed to deduct points');
        }
    },

    /**
     * Add points to user
     */
    async addPoints(userId, amount) {
        if (!this.isAdmin()) {
            throw new Error('Admin access required');
        }

        try {
            const userRef = db.collection('users').doc(userId);
            const userDoc = await userRef.get();

            if (!userDoc.exists) {
                throw new Error('User not found');
            }

            const currentPoints = userDoc.data().points || 0;
            const newPoints = currentPoints + amount;

            await userRef.update({
                points: newPoints,
                updatedAt: new Date().toISOString()
            });

            console.log(`✅ Added ${amount} points to user ${userId}`);
            return true;

        } catch (error) {
            console.error('❌ Error adding points:', error);
            throw error;
        }
    },

    /**
     * Login with email and password
     */
    async login(email, password) {
        try {
            const result = await fireAuth.signInWithEmailAndPassword(email, password);
            console.log('✅ Login successful');
            return result.user;
        } catch (error) {
            console.error('❌ Login error:', error);
            throw this.getErrorMessage(error.code);
        }
    },

    /**
     * Register new user
     */
    async register(email, password, displayName) {
        try {
            const result = await fireAuth.createUserWithEmailAndPassword(email, password);

            // Update profile
            await result.user.updateProfile({
                displayName: displayName
            });

            console.log('✅ Registration successful');
            return result.user;

        } catch (error) {
            console.error('❌ Registration error:', error);
            throw this.getErrorMessage(error.code);
        }
    },

    /**
     * Logout
     */
    async logout() {
        try {
            await fireAuth.signOut();
            this.currentUser = null;
            localStorage.removeItem('currentUser');
            console.log('✅ Logout successful');
            return true;
        } catch (error) {
            console.error('❌ Logout error:', error);
            throw new Error('فشل تسجيل الخروج');
        }
    },

    /**
     * Get all users (admin only)
     */
    async getAllUsers() {
        if (!this.isAdmin()) {
            throw new Error('Admin access required');
        }

        try {
            const snapshot = await db.collection('users').orderBy('createdAt', 'desc').get();
            return snapshot.docs.map(doc => ({
                id: doc.id,
                ...doc.data()
            }));
        } catch (error) {
            console.error('❌ Error fetching users:', error);
            throw error;
        }
    },

    /**
     * Update user data (admin only)
     */
    async updateUser(userId, updates) {
        if (!this.isAdmin()) {
            throw new Error('Admin access required');
        }

        try {
            await db.collection('users').doc(userId).update({
                ...updates,
                updatedAt: new Date().toISOString()
            });
            return true;
        } catch (error) {
            console.error('❌ Error updating user:', error);
            throw error;
        }
    },

    /**
     * Get user-friendly error messages
     */
    getErrorMessage(errorCode) {
        const errors = {
            'auth/invalid-email': 'البريد الإلكتروني غير صالح',
            'auth/user-disabled': 'تم تعطيل هذا الحساب',
            'auth/user-not-found': 'البريد الإلكتروني أو كلمة المرور غير صحيحة',
            'auth/wrong-password': 'البريد الإلكتروني أو كلمة المرور غير صحيحة',
            'auth/email-already-in-use': 'البريد الإلكتروني مستخدم بالفعل',
            'auth/weak-password': 'كلمة المرور ضعيفة. يجب أن تكون 6 أحرف على الأقل',
            'auth/network-request-failed': 'خطأ في الاتصال بالإنترنت',
            'auth/too-many-requests': 'محاولات كثيرة. الرجاء المحاولة لاحقاً'
        };

        return errors[errorCode] || 'حدث خطأ غير متوقع';
    },

    /**
     * Show error message to user
     */
    showError(message) {
        // Try to find error container
        const errorEl = document.getElementById('authError') ||
            document.querySelector('.alert-danger');

        if (errorEl) {
            errorEl.textContent = message;
            errorEl.classList.remove('d-none');
            errorEl.style.display = 'block';

            // Auto-hide after 5 seconds
            setTimeout(() => {
                errorEl.classList.add('d-none');
                errorEl.style.display = 'none';
            }, 5000);
        } else {
            // Fallback to alert
            alert(message);
        }
    }
};

// ============================================
// AUTO-INITIALIZE ON PAGE LOAD
// ============================================

document.addEventListener('DOMContentLoaded', async () => {
    console.log('🔐 Initializing authentication system...');

    try {
        await auth.init();
        console.log('✅ Authentication system ready');
    } catch (error) {
        console.error('❌ Failed to initialize auth:', error);
    }
});

// Export for use in other scripts
window.auth = auth;
