// ============================================
// MAIN APPLICATION LOGIC v2.0
// Modern, modular, memory-optimized
// ============================================

console.log('🚀 App.js v2.0 loading...');

// ============================================
// STATE MANAGEMENT
// ============================================

const appState = {
    currentResults: [],
    selectedPlaces: new Set(),
    searchInProgress: false,
    currentUser: null
};

// ============================================
// DOM ELEMENTS
// ============================================

const elements = {
    // Forms
    searchForm: null,
    keywordInput: null,
    locationInput: null,
    resultCountInput: null,

    // Results
    resultsSection: null,
    resultsGrid: null,
    resultsCount: null,
    searchError: null,

    // User
    userPoints: null,
    userName: null,
    adminBtn: null,

    // Loading
    searchSpinner: null,
    loadingSkeletons: null
};

// ============================================
// INITIALIZATION
// ============================================

document.addEventListener('DOMContentLoaded', async () => {
    console.log('📱 DOM loaded, initializing app...');

    // Initialize DOM references
    initializeDOMElements();

    // Wait for auth system
    await waitForAuth();

    // Check authentication
    const user = auth.getCurrentUser();
    if (!user) {
        console.log('❌ No user logged in, redirecting to login');
        window.location.href = 'login.html';
        return;
    }

    appState.currentUser = user;

    // Update UI with user info
    updateUserInfo(user);

    // Setup event listeners
    setupEventListeners();

    console.log('✅ App initialized successfully');
});

/**
 * Initialize DOM element references
 */
function initializeDOMElements() {
    elements.searchForm = document.getElementById('searchForm');
    elements.keywordInput = document.getElementById('keyword');
    elements.locationInput = document.getElementById('location');
    elements.resultCountInput = document.getElementById('resultCount');

    elements.resultsSection = document.getElementById('resultsSection');
    elements.resultsGrid = document.getElementById('resultsGrid');
    elements.resultsCount = document.getElementById('resultsCount');
    elements.searchError = document.getElementById('searchError');

    elements.userPoints = document.getElementById('userPoints');
    elements.userName = document.getElementById('userName');
    elements.adminBtn = document.getElementById('adminBtn');

    elements.searchSpinner = document.getElementById('searchSpinner');
}

/**
 * Wait for auth system to initialize
 */
async function waitForAuth() {
    return new Promise((resolve) => {
        const checkAuth = setInterval(() => {
            if (window.auth && window.auth.initialized) {
                clearInterval(checkAuth);
                resolve();
            }
        }, 100);

        // Timeout after 10 seconds
        setTimeout(() => {
            clearInterval(checkAuth);
            resolve();
        }, 10000);
    });
}

/**
 * Update UI with user information
 */
function updateUserInfo(user) {
    if (elements.userName) {
        elements.userName.textContent = user.displayName || user.email.split('@')[0];
    }

    if (elements.userPoints) {
        const points = auth.isAdmin() ? 'غير محدود' : user.points || 0;
        elements.userPoints.textContent = points;
    }

    if (elements.adminBtn && auth.isAdmin()) {
        elements.adminBtn.style.display = 'block';
    }
}

/**
 * Setup event listeners
 */
function setupEventListeners() {
    if (elements.searchForm) {
        elements.searchForm.addEventListener('submit', handleSearch);
    }
}

// ============================================
// SEARCH FUNCTIONALITY
// ============================================

/**
 * Handle search form submission
 */
async function handleSearch(e) {
    e.preventDefault();

    if (appState.searchInProgress) {
        return;
    }

    try {
        appState.searchInProgress = true;
        hideError();
        showLoading(true);

        // Get form values
        const keyword = elements.keywordInput.value.trim();
        const location = elements.locationInput.value.trim();
        const resultCount = parseInt(elements.resultCountInput.value) || 20;

        // Validation
        if (!keyword || !location) {
            throw new Error('الرجاء إدخال الكلمة المفتاحية والموقع');
        }

        console.log(`🔍 Searching: "${keyword}" in "${location}" (${resultCount} results)`);

        // Deduct points (unless admin)
        if (!auth.isAdmin()) {
            const pointCost = 1;
            const currentPoints = auth.getPoints();

            if (currentPoints < pointCost) {
                throw new Error('رصيد النقاط غير كافٍ');
            }

            await auth.deductPoints(pointCost);
            updateUserInfo(appState.currentUser);
        }

        // Perform search
        const results = await performSearch(keyword, location, resultCount);

        // Display results
        displayResults(results);

        showToast('success', `تم العثور على ${results.length} نتيجة`);

    } catch (error) {
        console.error('❌ Search error:', error);
        showError(error.message || 'حدث خطأ أثناء البحث');
        showToast('error', error.message || 'فشل البحث');
    } finally {
        appState.searchInProgress = false;
        showLoading(false);
    }
}

/**
 * Perform the actual search API call
 */
async function performSearch(keyword, location, resultCount) {
    const params = new URLSearchParams({
        keyword,
        location,
        resultCount
    });

    const response = await fetch(`/api/places/search?${params.toString()}`);

    if (!response.ok) {
        const error = await response.json();
        throw new Error(error.details || error.error || 'فشل البحث');
    }

    const data = await response.json();

    if (!data.results || data.results.length === 0) {
        throw new Error('لم يتم العثور على نتائج');
    }

    return data.results;
}

/**
 * Display search results
 */
function displayResults(results) {
    appState.currentResults = results;
    appState.selectedPlaces.clear();

    // Update count
    if (elements.resultsCount) {
        elements.resultsCount.textContent = results.length;
    }

    // Clear grid
    if (elements.resultsGrid) {
        elements.resultsGrid.innerHTML = '';

        // Create result cards
        results.forEach((place, index) => {
            const card = createResultCard(place, index);
            elements.resultsGrid.appendChild(card);
        });
    }

    // Show results section
    if (elements.resultsSection) {
        elements.resultsSection.classList.add('active');
        elements.resultsSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
}

/**
 * Create a result card element
 */
function createResultCard(place, index) {
    const card = document.createElement('div');
    card.className = 'result-card slide-up';
    card.style.animationDelay = `${index * 0.05}s`;

    const phone = place.formatted_phone_number || place.international_phone_number || 'غير متوفر';
    const address = place.formatted_address || place.vicinity || 'غير متوفر';
    const name = place.name || 'غير معروف';

    card.innerHTML = `
        <div class="result-card-header">
            <div>
                <h3 class="result-title">${name}</h3>
                <div class="result-address">
                    <i class="bi bi-geo-alt-fill"></i>
                    ${address}
                </div>
            </div>
            <input type="checkbox" class="form-check-input" onchange="toggleSelection(${index})">
        </div>
        
        <div class="result-phone">
            <i class="bi bi-telephone-fill"></i>
            ${phone}
        </div>
        
        <div class="result-actions">
            <button class="btn btn-sm btn-success" onclick="sendWhatsApp('${phone}', '${name}')">
                <i class="bi bi-whatsapp"></i>
                WhatsApp
            </button>
            
            <button class="btn btn-sm btn-outline" onclick="openMap('${place.place_id}')">
                <i class="bi bi-map"></i>
                خريطة
            </button>
        </div>
    `;

    return card;
}

// ============================================
// ACTIONS
// ============================================

/**
 * Toggle place selection
 */
function toggleSelection(index) {
    if (appState.selectedPlaces.has(index)) {
        appState.selectedPlaces.delete(index);
    } else {
        appState.selectedPlaces.add(index);
    }

    console.log(`Selected: ${appState.selectedPlaces.size} places`);
}

/**
 * Select all places
 */
function selectAll() {
    const checkboxes = elements.resultsGrid.querySelectorAll('input[type="checkbox"]');
    const allSelected = appState.selectedPlaces.size === appState.currentResults.length;

    if (allSelected) {
        // Deselect all
        appState.selectedPlaces.clear();
        checkboxes.forEach(cb => cb.checked = false);
        showToast('info', 'تم إلغاء تحديد الكل');
    } else {
        // Select all
        appState.selectedPlaces.clear();
        checkboxes.forEach((cb, i) => {
            cb.checked = true;
            appState.selectedPlaces.add(i);
        });
        showToast('success', `تم تحديد ${appState.selectedPlaces.size} نتيجة`);
    }
}

/**
 * Send WhatsApp message
 */
function sendWhatsApp(phone, name) {
    if (phone === 'غير متوفر' || !phone) {
        showToast('error', 'رقم الهاتف غير متوفر');
        return;
    }

    // Clean phone number
    const cleanPhone = phone.replace(/[^\d+]/g, '');

    // Default message
    const message = `السلام عليكم ${name}،\n\nأود التواصل معكم بخصوص خدماتكم.`;

    // Open WhatsApp
    const whatsappUrl = `https://wa.me/${cleanPhone}?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');

    showToast('success', 'تم فتح WhatsApp');
}

/**
 * Open place in Google Maps
 */
function openMap(placeId) {
    if (!placeId) {
        showToast('error', 'معرّف المكان غير متوفر');
        return;
    }

    const mapsUrl = `https://www.google.com/maps/place/?q=place_id:${placeId}`;
    window.open(mapsUrl, '_blank');
}

/**
 * Export results to CSV
 */
function exportResults() {
    if (!appState.currentResults || appState.currentResults.length === 0) {
        showToast('error', 'لا توجد نتائج للتصدير');
        return;
    }

    // Get selected or all results
    const toExport = appState.selectedPlaces.size > 0
        ? appState.currentResults.filter((_, i) => appState.selectedPlaces.has(i))
        : appState.currentResults;

    // Create CSV
    const headers = ['الاسم', 'العنوان', 'رقم الهاتف'];
    const rows = toExport.map(place => [
        place.name || '',
        place.formatted_address || place.vicinity || '',
        place.formatted_phone_number || place.international_phone_number || ''
    ]);

    const csv = [
        headers.join(','),
        ...rows.map(row => row.map(cell => `"${cell}"`).join(','))
    ].join('\n');

    // Add BOM for Excel UTF-8 support
    const bom = '\uFEFF';
    const blob = new Blob([bom + csv], { type: 'text/csv;charset=utf-8;' });

    // Download
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = `results_${Date.now()}.csv`;
    link.click();

    showToast('success', `تم تصدير ${toExport.length} نتيجة`);
}

/**
 * Clear search form
 */
function clearSearch() {
    elements.keywordInput.value = '';
    elements.locationInput.value = '';
    elements.resultCountInput.value = '20';
    appState.currentResults = [];
    appState.selectedPlaces.clear();

    if (elements.resultsSection) {
        elements.resultsSection.classList.remove('active');
    }

    hideError();
}

/**
 * Handle logout
 */
async function handleLogout() {
    try {
        await auth.logout();
        window.location.href = 'login.html';
    } catch (error) {
        console.error('Logout error:', error);
        showToast('error', 'فشل تسجيل الخروج');
    }
}

// ============================================
// UI HELPERS
// ============================================

/**
 * Show/hide loading state
 */
function showLoading(show) {
    if (elements.searchSpinner) {
        elements.searchSpinner.style.display = show ? 'inline-block' : 'none';
    }

    // Disable form
    const submitBtn = elements.searchForm?.querySelector('button[type="submit"]');
    if (submitBtn) {
        submitBtn.disabled = show;
    }
}

/**
 * Show error message
 */
function showError(message) {
    if (elements.searchError) {
        elements.searchError.textContent = message;
        elements.searchError.classList.remove('d-none');
        elements.searchError.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
}

/**
 * Hide error message
 */
function hideError() {
    if (elements.searchError) {
        elements.searchError.classList.add('d-none');
    }
}

/**
 * Show toast notification
 */
function showToast(type, message) {
    const container = document.getElementById('toastContainer');
    if (!container) return;

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    const icon = type === 'success' ? 'check-circle-fill' :
        type === 'error' ? 'exclamation-triangle-fill' :
            'info-circle-fill';

    toast.innerHTML = `
        <i class="bi bi-${icon}"></i>
        <span>${message}</span>
    `;

    container.appendChild(toast);

    // Auto remove after 3 seconds
    setTimeout(() => {
        toast.style.opacity = '0';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Export functions for onclick handlers
window.toggleSelection = toggleSelection;
window.selectAll = selectAll;
window.sendWhatsApp = sendWhatsApp;
window.openMap = openMap;
window.exportResults = exportResults;
window.clearSearch = clearSearch;
window.handleLogout = handleLogout;

console.log('✅ App.js v2.0 loaded successfully');
