/**
 * admin.js - وظائف لوحة تحكم المدير
 * 
 * يتضمن وظائف إدارة المستخدمين، نظام النقاط، وعرض إحصائيات النظام
 */

// المتغيرات العامة
let systemSettings = {
    pointsPerResult: 1,
    defaultPointsForNewUser: 50
};

// تحميل إعدادات النظام عند تحميل الصفحة
// لتخزين حالة التصفية الحالية
let currentEmailFilter = 'all';
let currentSearchQuery = '';

// Expose functions to global scope for admin-panel.html
window.loadUsers = loadUsers;
window.loadPendingAccounts = loadPendingAccounts;
window.updateSystemStats = () => { if (window.loadStatistics) window.loadStatistics(); };

document.addEventListener('DOMContentLoaded', () => {
    // الحصول على مراجع عناصر DOM
    const usersTableBody = document.getElementById('usersTableBody');
    const pendingAccountsTableBody = document.getElementById('pendingAccountsTableBody');
    const userSelect = document.getElementById('userSelect');
    const refreshUsersBtn = document.getElementById('refreshUsersBtn');
    const saveUserChangesBtn = document.getElementById('saveUserChangesBtn');
    const createUserSubmitBtn = document.getElementById('createUserSubmitBtn');

    // عناصر البحث والتصفية
    const searchUserInput = document.getElementById('searchUserInput');
    const searchUserBtn = document.getElementById('searchUserBtn');
    const showAllEmailsBtn = document.getElementById('showAllEmailsBtn');
    const showPersonalEmailsBtn = document.getElementById('showPersonalEmailsBtn');
    const showCompanyEmailsBtn = document.getElementById('showCompanyEmailsBtn');

    // تحميل إعدادات النظام من localStorage أو استخدام القيم الافتراضية
    loadSystemSettings();

    // تحميل بيانات المستخدمين
    loadUsers();

    // تحميل الحسابات التي تنتظر التفعيل
    loadPendingAccounts();

    // تحميل الإحصائيات
    loadStatistics();

    // إضافة مستمعي الأحداث للأزرار
    refreshUsersBtn.addEventListener('click', loadUsers);
    saveUserChangesBtn.addEventListener('click', saveUserChanges);

    // إضافة مستمع الحدث لزر إنشاء الحساب
    if (createUserSubmitBtn) {
        createUserSubmitBtn.addEventListener('click', createNewUser);
    }

    // إضافة مستمعى الأحداث لأزرار البحث والتصفية
    if (searchUserBtn && searchUserInput) {
        // زر البحث
        searchUserBtn.addEventListener('click', () => {
            currentSearchQuery = searchUserInput.value.trim().toLowerCase();
            loadUsers();
        });

        // البحث عند الضغط على زر Enter
        searchUserInput.addEventListener('keyup', (event) => {
            if (event.key === 'Enter') {
                currentSearchQuery = searchUserInput.value.trim().toLowerCase();
                loadUsers();
            }
        });
    }

    // أزرار تصفية البريد الإلكتروني
    if (showAllEmailsBtn && showPersonalEmailsBtn && showCompanyEmailsBtn) {
        // عرض جميع المستخدمين
        showAllEmailsBtn.addEventListener('click', () => {
            setActiveEmailFilterButton(showAllEmailsBtn);
            currentEmailFilter = 'all';
            loadUsers();
        });

        // عرض الحسابات الشخصية فقط
        showPersonalEmailsBtn.addEventListener('click', () => {
            setActiveEmailFilterButton(showPersonalEmailsBtn);
            currentEmailFilter = 'personal';
            loadUsers();
        });

        // عرض حسابات الشركات فقط
        showCompanyEmailsBtn.addEventListener('click', () => {
            setActiveEmailFilterButton(showCompanyEmailsBtn);
            currentEmailFilter = 'company';
            loadUsers();
        });
    }

    // إضافة مستمعي الأحداث لنماذج الإدخال
    document.getElementById('pointsSettingsForm').addEventListener('submit', saveSystemSettings);
    document.getElementById('addPointsForm').addEventListener('submit', addPointsToUser);
    const apiSettingsForm = document.getElementById('apiSettingsForm');
    if (apiSettingsForm) apiSettingsForm.addEventListener('submit', saveApiSettings);

    // إضافة مستمعي الأحداث لعلامات تبويب القائمة
    document.getElementById('nav-users-tab').addEventListener('click', showSection.bind(null, 'users'));
    document.getElementById('nav-points-tab').addEventListener('click', showSection.bind(null, 'points'));
    document.getElementById('nav-stats-tab').addEventListener('click', showSection.bind(null, 'stats'));
    const navApi = document.getElementById('nav-api-tab');
    if (navApi) navApi.addEventListener('click', showSection.bind(null, 'api'));
});

/**
 * عرض قسم محدد وإخفاء الأقسام الأخرى
 * @param {string} sectionName - اسم القسم المراد عرضه
 * @param {Event} event - حدث النقر
 */
function showSection(sectionName, event) {
    if (event) {
        event.preventDefault();

        // إزالة الفئة النشطة من جميع علامات التبويب
        document.querySelectorAll('.list-group-item').forEach(tab => {
            tab.classList.remove('active');
        });

        // إضافة الفئة النشطة إلى علامة التبويب المحددة
        event.target.classList.add('active');
    }

    // إخفاء جميع الأقسام
    document.querySelectorAll('.admin-section').forEach(section => {
        section.classList.add('d-none');
    });

    // عرض القسم المحدد
    document.getElementById(`${sectionName}-section`).classList.remove('d-none');
    if (sectionName === 'api') loadApiSettings();
}

/**
 * التحقق مما إذا كان البريد الإلكتروني تابع لشركة أم حساب شخصي
 * @param {string} email - البريد الإلكتروني للتحقق منه
 * @returns {string} - 'شركة' أو 'شخصي'
 */
function getEmailType(email) {
    if (!email) return 'شخصي';

    // قائمة بالمجالات الشائعة للبريد الإلكتروني الشخصي
    const personalDomains = [
        'gmail.com', 'yahoo.com', 'hotmail.com', 'outlook.com', 'aol.com',
        'icloud.com', 'mail.com', 'protonmail.com', 'yandex.com', 'zoho.com',
        'inbox.com', 'gmx.com', 'live.com', 'msn.com', 'me.com'
    ];

    // فحص مجال البريد الإلكتروني
    const domain = email.toLowerCase().split('@')[1];

    if (personalDomains.includes(domain)) {
        return 'شخصي';
    } else {
        return 'شركة';
    }
}

/**
 * تعيين الزر النشط في مجموعة تصفية البريد الإلكتروني
 * @param {HTMLElement} activeButton - الزر الذي سيكون نشطًا
 */
function setActiveEmailFilterButton(activeButton) {
    // إزالة الفئة النشطة من جميع الأزرار
    document.getElementById('showAllEmailsBtn').classList.remove('active');
    document.getElementById('showPersonalEmailsBtn').classList.remove('active');
    document.getElementById('showCompanyEmailsBtn').classList.remove('active');

    // إضافة الفئة النشطة للزر المحدد
    activeButton.classList.add('active');

    // تحديث نص حالة التصفية الحالية
    const currentFilterTextElement = document.getElementById('currentFilterText');
    if (currentFilterTextElement) {
        // تحديد نص التصفية بناءً على الزر النشط
        let filterText = 'جميع الحسابات';

        const dataType = activeButton.getAttribute('data-email-type');
        if (dataType === 'personal') {
            filterText = 'الحسابات الشخصية فقط';
        } else if (dataType === 'company') {
            filterText = 'حسابات الشركات فقط';
        }

        currentFilterTextElement.textContent = filterText;
    }
}

/**
 * تحميل بيانات المستخدمين وعرضها في الجدول
 */
async function loadUsers() {
    console.log('تحميل بيانات المستخدمين...');
    console.log('حالة التصفية:', currentEmailFilter);
    console.log('استعلام البحث:', currentSearchQuery);

    const usersTableBody = document.getElementById('usersTableBody');
    if (!usersTableBody) return;

    // الحصول على قائمة المستخدمين من فايربيس
    const snapshot = await db.collection('users').get();
    const allUsers = snapshot.docs.map(doc => ({ id: doc.id, ...doc.data() }));

    // تطبيق التصفية والبحث
    const filteredUsers = allUsers.filter(user => {
        // تطبيق تصفية نوع البريد الإلكتروني
        if (currentEmailFilter !== 'all') {
            const userEmailType = getEmailType(user.email);
            if (currentEmailFilter === 'company' && userEmailType !== 'شركة') {
                return false;
            } else if (currentEmailFilter === 'personal' && userEmailType !== 'شخصي') {
                return false;
            }
        }

        // تطبيق البحث
        if (currentSearchQuery) {
            const nameMatch = user.name && user.name.toLowerCase().includes(currentSearchQuery);
            const emailMatch = user.email && user.email.toLowerCase().includes(currentSearchQuery);
            return nameMatch || emailMatch;
        }

        return true;
    });

    // مسح محتوى الجدول
    usersTableBody.innerHTML = '';

    if (filteredUsers.length === 0) {
        // إذا لم تكن هناك بيانات للمستخدمين
        const emptyRow = document.createElement('tr');
        emptyRow.innerHTML = `
            <td colspan="8" class="text-center">لا توجد بيانات مستخدمين مطابقة</td>
        `;
        usersTableBody.appendChild(emptyRow);
        return;
    }

    // تحديث قائمة المستخدمين في قسم إضافة النقاط
    const userSelect = document.getElementById('userSelect');
    if (userSelect) {
        userSelect.innerHTML = '<option value="">اختر مستخدمًا...</option>';

        filteredUsers.forEach(user => {
            const option = document.createElement('option');
            option.value = user.id;
            option.textContent = `${user.name} (${user.email})`;
            userSelect.appendChild(option);
        });
    }

    // عرض بيانات كل مستخدم في الجدول
    filteredUsers.forEach(user => {
        // تنسيق تاريخ آخر تسجيل دخول
        const lastLogin = user.lastLogin ? new Date(user.lastLogin).toLocaleString('ar-SA') : 'لم يسجل بعد';

        // تحديد نوع الحساب (شخصي أو شركة)
        const emailType = getEmailType(user.email);
        const accountTypeBadge = emailType === 'شركة'
            ? `<span class="badge bg-primary">شركة</span>`
            : `<span class="badge bg-secondary">شخصي</span>`;

        // تنسيق تاريخ التسجيل
        const createdAt = user.createdAt || user.registrationDate;
        const registrationDate = createdAt ? new Date(createdAt).toLocaleString('ar-SA') : 'غير معروف';

        // إنشاء صف للمستخدم في الجدول
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${user.name || 'بدون اسم'}</td>
            <td>${user.email || 'بدون بريد'}</td>
            <td>${accountTypeBadge}</td>
            <td>${user.points || 0}</td>
            <td>${registrationDate}</td>
            <td>${lastLogin}</td>
            <td>
                <span class="badge ${user.isActive ? 'bg-success' : 'bg-danger'}">
                    ${user.isActive ? 'نشط' : 'معلق'}
                </span>
            </td>
            <td>
                <div class="btn-group btn-group-sm">
                    <button class="btn btn-primary edit-user-btn" data-user-id="${user.id}">
                        <i class="bi bi-pencil"></i> تعديل
                    </button>
                    <button class="btn ${user.isActive ? 'btn-danger' : 'btn-success'} toggle-user-status-btn" data-user-id="${user.id}" data-status="${!user.isActive}">
                        <i class="bi ${user.isActive ? 'bi-x-circle' : 'bi-check-circle'}"></i> ${user.isActive ? 'تعليق' : 'تفعيل'}
                    </button>
                    <button class="btn btn-info device-info-btn" data-user-id="${user.id}">
                        <i class="bi bi-info-circle"></i> معلومات الجهاز
                    </button>
                    <button class="btn btn-secondary user-searches-btn" data-user-id="${user.id}">
                        <i class="bi bi-search"></i> عمليات البحث
                    </button>
                </div>
            </td>
        `;
        usersTableBody.appendChild(row);

        // إضافة المستخدم إلى قائمة الاختيار في نموذج إضافة النقاط
        const option = document.createElement('option');
        option.value = user.id;
        option.textContent = `${user.name} (${user.email})`;
        userSelect.appendChild(option);
    });

    // إضافة مستمعي الأحداث لأزرار الإجراءات
    addActionButtonListeners();
}

/**
 * تحميل الحسابات التي تنتظر التفعيل
 */
function loadPendingAccounts() {
    console.log('تحميل الحسابات التي تنتظر التفعيل...');

    const pendingAccountsTableBody = document.getElementById('pendingAccountsTableBody');
    if (!pendingAccountsTableBody) {
        console.error('لم يتم العثور على عنصر pendingAccountsTableBody!');
        return;
    }

    const pendingAccountsCount = document.getElementById('pendingAccountsCount');
    const noPendingAccountsMessage = document.getElementById('noPendingAccountsMessage');

    // الحصول على البيانات المخزنة مباشرة من localStorage
    const usersJson = localStorage.getItem('users');
    console.log('البيانات الخام من localStorage:', usersJson);

    // الحصول على قائمة المستخدمين
    const users = JSON.parse(usersJson || '[]');
    console.log('إجمالي عدد المستخدمين المسجلين:', users.length);

    // عرض بيانات جميع المستخدمين للتشخيص
    console.log('جميع المستخدمين:', users);

    // تصفية المستخدمين الذين ينتظرون التفعيل - نتحقق من قيمتي activationPending و isActive
    const pendingUsers = users.filter(user => {
        // يعتبر المستخدم في انتظار التفعيل إذا كان activationPending = true أو isActive = false
        const isPending = user.activationPending === true || (user.isActive === false && user.isAdmin !== true);
        console.log(`المستخدم ${user.name} (${user.email}): activationPending = ${user.activationPending}, isActive = ${user.isActive}, isPending = ${isPending}`);
        return isPending;
    });

    console.log('عدد المستخدمين الذين ينتظرون التفعيل:', pendingUsers.length);
    console.log('المستخدمون الذين ينتظرون التفعيل:', pendingUsers);

    // مسح محتوى الجدول
    pendingAccountsTableBody.innerHTML = '';

    // تحديث عدد الحسابات التي تنتظر التفعيل
    if (pendingAccountsCount) {
        pendingAccountsCount.textContent = pendingUsers.length;
    }

    // تحديث عنوان القسم بعدد الحسابات التي تنتظر التفعيل
    const pendingAccountsBadge = document.getElementById('pendingAccountsBadge');
    if (pendingAccountsBadge) {
        pendingAccountsBadge.textContent = pendingUsers.length;
        pendingAccountsBadge.style.display = pendingUsers.length > 0 ? 'inline-block' : 'none';
    }

    // إظهار رسالة إذا لم تكن هناك حسابات تنتظر التفعيل
    if (pendingUsers.length === 0) {
        if (noPendingAccountsMessage) {
            noPendingAccountsMessage.classList.remove('d-none');
        } else {
            // إذا لم يكن هناك عنصر للرسالة، نضيف رسالة مباشرة إلى الجدول
            const emptyRow = document.createElement('tr');
            emptyRow.innerHTML = `
                <td colspan="4" class="text-center">لا توجد حسابات تنتظر التفعيل حاليًا</td>
            `;
            pendingAccountsTableBody.appendChild(emptyRow);
        }
        return;
    } else if (noPendingAccountsMessage) {
        noPendingAccountsMessage.classList.add('d-none');
    }

    // عرض بيانات كل مستخدم في الجدول
    pendingUsers.forEach(user => {
        // تنسيق تاريخ التسجيل
        const createdAt = user.registrationDate ? new Date(user.registrationDate).toLocaleString('ar-SA') : 'غير معروف';

        // إضافة صف للمستخدم في جدول الحسابات التي تنتظر التفعيل
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${user.name}</td>
            <td>${user.email}</td>
            <td>${createdAt}</td>
            <td>
                <div class="btn-group btn-group-sm">
                    <button class="btn btn-success activate-user-btn" data-user-id="${user.id}">
                        <i class="bi bi-check-circle"></i> تفعيل
                    </button>
                    <button class="btn btn-danger reject-user-btn" data-user-id="${user.id}">
                        <i class="bi bi-x-circle"></i> رفض
                    </button>
                </div>
            </td>
        `;
        pendingAccountsTableBody.appendChild(row);
    });

    // إضافة مستمعي الأحداث لأزرار التفعيل والرفض
    addPendingAccountsButtonListeners();
}

/**
 * إضافة مستمعي الأحداث لأزرار التفعيل والرفض للحسابات التي تنتظر التفعيل
 */
function addPendingAccountsButtonListeners() {
    // أزرار تفعيل الحساب
    document.querySelectorAll('.activate-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            activateUserAccount(userId);
        });
    });

    // أزرار رفض الحساب
    document.querySelectorAll('.reject-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            rejectUserAccount(userId);
        });
    });
}

/**
 * تفعيل حساب مستخدم
 * @param {string} userId - معرف المستخدم
 */
function activateUserAccount(userId) {
    console.log(`محاولة تفعيل حساب المستخدم بمعرف: ${userId}`);

    // الحصول على قائمة المستخدمين مباشرة من localStorage لضمان الحصول على أحدث البيانات
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    console.log(`تم العثور على ${users.length} مستخدمين في قاعدة البيانات`);

    // البحث عن المستخدم
    const userIndex = users.findIndex(u => u.id === userId);

    if (userIndex === -1) {
        console.error(`خطأ: لم يتم العثور على المستخدم بمعرف ${userId}`);
        return;
    }

    console.log(`تم العثور على المستخدم:`, users[userIndex]);

    // تفعيل الحساب
    users[userIndex].activationPending = false;
    users[userIndex].isActive = true;

    // إضافة نقاط افتراضية للمستخدم الجديد
    if (!users[userIndex].points || users[userIndex].points === 0) {
        // الحصول على النقاط الافتراضية من إعدادات النظام
        const systemSettings = localStorage.getItem('systemSettings') ?
            JSON.parse(localStorage.getItem('systemSettings')) :
            { defaultPointsForNewUser: 50 };

        users[userIndex].points = systemSettings.defaultPointsForNewUser || 50;
        console.log(`تم إعطاء المستخدم ${users[userIndex].points} نقطة`);
    }

    // حفظ التغييرات
    try {
        localStorage.setItem('users', JSON.stringify(users));
        console.log(`تم تحديث بيانات المستخدمين بنجاح. المستخدم ${users[userIndex].name} أصبح نشطًا الآن.`);

        // تحديث العرض
        loadPendingAccounts();
        loadUsers();

        // إظهار رسالة نجاح
        showSuccessMessage(`تم تفعيل حساب المستخدم ${users[userIndex].name} بنجاح`);
    } catch (error) {
        console.error('خطأ في حفظ بيانات المستخدمين:', error);
        alert('حدث خطأ أثناء تفعيل الحساب');
    }
}

/**
 * رفض حساب مستخدم
 * @param {string} userId - معرف المستخدم
 */
function rejectUserAccount(userId) {
    console.log(`محاولة رفض حساب المستخدم بمعرف: ${userId}`);

    // الحصول على قائمة المستخدمين مباشرة من localStorage لضمان الحصول على أحدث البيانات
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    console.log(`تم العثور على ${users.length} مستخدمين في قاعدة البيانات`);

    // البحث عن المستخدم
    const userIndex = users.findIndex(u => u.id === userId);

    if (userIndex === -1) {
        console.error(`خطأ: لم يتم العثور على المستخدم بمعرف ${userId}`);
        return;
    }

    console.log(`تم العثور على المستخدم:`, users[userIndex]);

    // حفظ اسم المستخدم لعرضه في رسالة النجاح
    const userName = users[userIndex].name || 'غير معروف';
    const userEmail = users[userIndex].email || 'غير معروف';

    console.log(`جاري رفض حساب المستخدم ${userName} (${userEmail})`);

    // حذف المستخدم من المصفوفة
    users.splice(userIndex, 1);

    // حفظ التغييرات
    try {
        localStorage.setItem('users', JSON.stringify(users));
        console.log(`تم تحديث بيانات المستخدمين بنجاح. تم رفض وحذف حساب المستخدم ${userName}.`);

        // تحديث العرض
        loadPendingAccounts();
        loadUsers();

        // إظهار رسالة نجاح
        showSuccessMessage(`تم رفض حساب المستخدم ${userName} بنجاح`);
    } catch (error) {
        console.error('خطأ في حفظ بيانات المستخدمين:', error);
        alert('حدث خطأ أثناء رفض الحساب');
    }
}

/**
 * إظهار رسالة نجاح
 * @param {string} message - نص الرسالة
 */
function showSuccessMessage(message) {
    // التحقق من وجود عنصر التنبيه
    const alertContainer = document.getElementById('alertContainer');
    if (!alertContainer) {
        // إذا لم يكن هناك حاوية تنبيهات، استخدم وظيفة auth.showSuccessAlert
        auth.showSuccessAlert(message);
        return;
    }

    // إنشاء عنصر التنبيه
    const alert = document.createElement('div');
    alert.className = 'alert alert-success alert-dismissible fade show';
    alert.setAttribute('role', 'alert');
    alert.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="إغلاق"></button>
    `;

    // إضافة التنبيه إلى الحاوية
    alertContainer.appendChild(alert);

    // إزالة التنبيه بعد 5 ثوانٍ
    setTimeout(() => {
        alert.classList.remove('show');
        setTimeout(() => {
            if (alertContainer.contains(alert)) {
                alertContainer.removeChild(alert);
            }
        }, 300);
    }, 5000);
}

/**
 * إضافة مستمعي الأحداث لأزرار الإجراءات في جدول المستخدمين
 */
function addActionButtonListeners() {
    // أزرار تعديل بيانات المستخدم
    document.querySelectorAll('.edit-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            openEditUserModal(userId);
        });
    });

    // أزرار عرض معلومات الجهاز
    document.querySelectorAll('.view-device-info-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            openDeviceInfoModal(userId);
        });
    });

    // أزرار عرض عمليات البحث الخاصة بالمستخدم
    document.querySelectorAll('.view-searches-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            openUserSearchesModal(userId);
        });
    });

    // أزرار حظر/إلغاء حظر المستخدم
    document.querySelectorAll('.block-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            toggleUserStatus(userId, false);
        });
    });

    document.querySelectorAll('.unblock-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            toggleUserStatus(userId, true);
        });
    });
}

/**
 * فتح نافذة تعديل بيانات المستخدم
 * @param {string} userId - معرف المستخدم
 */
function openEditUserModal(userId) {
    // الحصول على بيانات المستخدم
    const user = auth.getUsers().find(u => u.id === userId);

    if (!user) {
        alert('المستخدم غير موجود!');
        return;
    }

    // ملء حقول النموذج ببيانات المستخدم
    document.getElementById('editUserId').value = user.id;
    document.getElementById('editUserName').value = user.name;
    document.getElementById('editUserEmail').value = user.email;
    document.getElementById('editUserPoints').value = user.points || 0;
    document.getElementById('editUserActive').checked = user.isActive !== false;
    document.getElementById('editUserAdmin').checked = user.isAdmin === true;
    document.getElementById('editWhatsappToken').value = user.whatsappToken || '';
    document.getElementById('editWhatsappPhoneId').value = user.whatsappPhoneNumberId || '';
    document.getElementById('editWhatsappDefaultCountry').value = user.whatsappDefaultCountryCode || '966';
    document.getElementById('editWhatsappUseAi').checked = (user.whatsappUseAi ?? true);
    document.getElementById('editUseWhatsappQr').checked = (user.useWhatsappQr ?? false);
    document.getElementById('editWhatsappTemplate').value = user.whatsappTemplate || '';
    document.getElementById('editWhatsappDelayMs').value = user.whatsappDelayMs || 2000;
    document.getElementById('editGoogleMapsApiKey').value = user.googleMapsApiKey || '';
    document.getElementById('editAiProvider').value = (user.aiProvider || 'openai');
    document.getElementById('editAiApiKey').value = user.aiApiKey || '';
    document.getElementById('editAiModel').value = user.aiModel || '';

    // فتح النافذة
    const modal = new bootstrap.Modal(document.getElementById('editUserModal'));
    modal.show();
}

/**
 * فتح نافذة معلومات جهاز المستخدم
 * @param {string} userId - معرف المستخدم
 */
function openDeviceInfoModal(userId) {
    // الحصول على بيانات المستخدم
    const users = auth.getUsers();
    const user = users.find(u => u.id === userId);

    if (!user) return;

    // الحصول على مراجع عناصر النافذة
    const deviceInfoModal = new bootstrap.Modal(document.getElementById('deviceInfoModal'));
    const deviceInfoTitle = document.getElementById('deviceInfoTitle');
    const deviceInfoContent = document.getElementById('deviceInfoContent');

    // تعيين عنوان النافذة
    deviceInfoTitle.textContent = `معلومات جهاز ${user.name}`;

    // تحضير محتوى النافذة
    let content = '<div class="table-responsive"><table class="table table-bordered">';
    content += '<tr><th>المتصفح</th><td>' + (user.deviceInfo?.browser || 'غير معروف') + '</td></tr>';
    content += '<tr><th>نظام التشغيل</th><td>' + (user.deviceInfo?.os || 'غير معروف') + '</td></tr>';
    content += '<tr><th>الجهاز</th><td>' + (user.deviceInfo?.device || 'غير معروف') + '</td></tr>';
    content += '<tr><th>الشاشة</th><td>' + (user.deviceInfo?.screen || 'غير معروف') + '</td></tr>';
    content += '<tr><th>IP</th><td>' + (user.deviceInfo?.ip || 'غير معروف') + '</td></tr>';
    content += '<tr><th>آخر تسجيل دخول</th><td>' + (user.lastLogin ? new Date(user.lastLogin).toLocaleString('ar-SA') : 'غير معروف') + '</td></tr>';
    content += '</table></div>';

    // تعيين محتوى النافذة
    deviceInfoContent.innerHTML = content;

    // عرض النافذة
    deviceInfoModal.show();
}

/**
 * فتح نافذة عمليات البحث الخاصة بالمستخدم
 * @param {string} userId - معرف المستخدم
 */
function openUserSearchesModal(userId) {
    // الحصول على بيانات المستخدم
    const users = auth.getUsers();
    const user = users.find(u => u.id === userId);

    if (!user) return;

    // الحصول على مراجع عناصر النافذة
    const userSearchesModal = new bootstrap.Modal(document.getElementById('userSearchesModal'));
    const userSearchesTitle = document.getElementById('userSearchesTitle');
    const userSearchesContent = document.getElementById('userSearchesContent');

    // تعيين عنوان النافذة
    userSearchesTitle.textContent = `عمليات البحث الخاصة بـ ${user.name}`;

    // الحصول على عمليات البحث الخاصة بالمستخدم
    const searches = user.savedSearches || [];

    // تحضير محتوى النافذة
    let content = '';

    if (searches.length === 0) {
        content = '<div class="alert alert-info">لا توجد عمليات بحث مسجلة لهذا المستخدم.</div>';
    } else {
        content = '<div class="table-responsive"><table class="table table-striped table-bordered">';
        content += '<thead><tr><th>التاريخ</th><th>الكلمة المفتاحية</th><th>الموقع</th><th>عدد النتائج</th><th>النقاط المخصومة</th></tr></thead>';
        content += '<tbody>';

        // ترتيب عمليات البحث حسب التاريخ من الأحدث إلى الأقدم
        const sortedSearches = [...searches].sort((a, b) => new Date(b.date) - new Date(a.date));

        sortedSearches.forEach(search => {
            const date = new Date(search.date).toLocaleString('ar-SA');
            content += `<tr>
                <td>${date}</td>
                <td>${search.keyword || 'غير محدد'}</td>
                <td>${search.location || 'غير محدد'}</td>
                <td>${search.resultCount || 0}</td>
                <td>${search.pointsDeducted || 0}</td>
            </tr>`;
        });

        content += '</tbody></table></div>';
    }

    // تعيين محتوى النافذة
    userSearchesContent.innerHTML = content;

    // عرض النافذة
    userSearchesModal.show();
}

/**
 * حفظ التغييرات على بيانات المستخدم
 */
function saveUserChanges() {
    // الحصول على قيم الحقول
    const userId = document.getElementById('editUserId').value;
    const name = document.getElementById('editUserName').value;
    const email = document.getElementById('editUserEmail').value;
    const points = parseInt(document.getElementById('editUserPoints').value) || 0;
    const isActive = document.getElementById('editUserActive').checked;
    const isAdmin = document.getElementById('editUserAdmin').checked;
    const whatsappToken = document.getElementById('editWhatsappToken').value;
    const whatsappPhoneId = document.getElementById('editWhatsappPhoneId').value;
    const whatsappDefaultCountry = document.getElementById('editWhatsappDefaultCountry').value;
    const whatsappUseAi = document.getElementById('editWhatsappUseAi').checked;
    const useWhatsappQr = document.getElementById('editUseWhatsappQr').checked;
    const whatsappTemplate = document.getElementById('editWhatsappTemplate').value;
    const whatsappDelayMs = parseInt(document.getElementById('editWhatsappDelayMs').value) || 2000;
    const googleMapsApiKey = document.getElementById('editGoogleMapsApiKey').value;
    const aiProvider = document.getElementById('editAiProvider').value;
    const aiApiKey = document.getElementById('editAiApiKey').value;
    const aiModel = document.getElementById('editAiModel').value;

    // التحقق من صحة البيانات
    if (!name || !email) {
        alert('يرجى ملء جميع الحقول المطلوبة');
        return;
    }

    // الحصول على قائمة المستخدمين
    let users = auth.getUsers();

    // التحقق من عدم وجود بريد إلكتروني مكرر (ما عدا للمستخدم نفسه)
    const duplicateEmail = users.find(u => u.email === email && u.id !== userId);
    if (duplicateEmail) {
        alert('البريد الإلكتروني مستخدم بالفعل');
        return;
    }

    // تحديث بيانات المستخدم
    const updatedUsers = users.map(user => {
        if (user.id === userId) {
            return {
                ...user,
                name,
                email,
                points,
                isActive,
                isAdmin,
                whatsappToken,
                whatsappPhoneNumberId: whatsappPhoneId,
                whatsappDefaultCountryCode: whatsappDefaultCountry,
                whatsappUseAi,
                useWhatsappQr,
                whatsappTemplate,
                whatsappDelayMs,
                googleMapsApiKey,
                aiProvider,
                aiApiKey,
                aiModel
            };
        }
        return user;
    });

    // حفظ التغييرات
    auth.saveUsers(updatedUsers);

    // إغلاق النافذة
    const modal = bootstrap.Modal.getInstance(document.getElementById('editUserModal'));
    modal.hide();

    // إعادة تحميل بيانات المستخدمين
    loadUsers();

    // عرض رسالة نجاح
    alert('تم حفظ التغييرات بنجاح');
}

/**
 * تغيير حالة المستخدم (نشط/محظور)
 * @param {string} userId - معرف المستخدم
 * @param {boolean} isActive - الحالة الجديدة للمستخدم
 */
function toggleUserStatus(userId, isActive) {
    // الحصول على قائمة المستخدمين
    let users = auth.getUsers();

    // تحديث حالة المستخدم
    const updatedUsers = users.map(user => {
        if (user.id === userId) {
            return {
                ...user,
                isActive
            };
        }
        return user;
    });

    // حفظ التغييرات
    auth.saveUsers(updatedUsers);

    // إعادة تحميل بيانات المستخدمين
    loadUsers();

    // عرض رسالة نجاح
    alert(`تم ${isActive ? 'تفعيل' : 'حظر'} المستخدم بنجاح`);
}

/**
 * إضافة نقاط لمستخدم
 * @param {Event} event - حدث النموذج
 */
function addPointsToUser(event) {
    event.preventDefault();

    // الحصول على قيم الحقول
    const userId = document.getElementById('userSelect').value;
    const pointsToAdd = parseInt(document.getElementById('pointsToAdd').value) || 0;

    // التحقق من صحة البيانات
    if (!userId) {
        alert('يرجى اختيار مستخدم');
        return;
    }

    if (pointsToAdd <= 0) {
        alert('يرجى إدخال عدد نقاط صحيح');
        return;
    }

    // الحصول على قائمة المستخدمين
    let users = auth.getUsers();

    // تحديث نقاط المستخدم
    const updatedUsers = users.map(user => {
        if (user.id === userId) {
            const currentPoints = user.points || 0;
            return {
                ...user,
                points: currentPoints + pointsToAdd
            };
        }
        return user;
    });

    // حفظ التغييرات
    auth.saveUsers(updatedUsers);

    // إعادة تحميل بيانات المستخدمين
    loadUsers();

    // عرض رسالة نجاح
    alert(`تم إضافة ${pointsToAdd} نقطة للمستخدم بنجاح`);

    // إعادة تعيين النموذج
    document.getElementById('addPointsForm').reset();
}

/**
 * تحميل إعدادات النظام
 */
function loadSystemSettings() {
    // الحصول على الإعدادات من localStorage أو استخدام القيم الافتراضية
    const savedSettings = localStorage.getItem('systemSettings');
    if (savedSettings) {
        systemSettings = JSON.parse(savedSettings);
    }

    // تحديث قيم النموذج
    document.getElementById('pointsPerResult').value = systemSettings.pointsPerResult;
    document.getElementById('defaultPointsForNewUser').value = systemSettings.defaultPointsForNewUser;
}

function loadApiSettings() {
    const savedSettings = localStorage.getItem('systemSettings');
    const s = savedSettings ? JSON.parse(savedSettings) : {};
    const mapsKeyEl = document.getElementById('systemGoogleMapsApiKey');
    const aiProviderEl = document.getElementById('systemAiProvider');
    const aiKeyEl = document.getElementById('systemAiApiKey');
    const aiModelEl = document.getElementById('systemAiModel');
    if (mapsKeyEl) mapsKeyEl.value = s.googleMapsApiKey || '';
    if (aiProviderEl) aiProviderEl.value = s.aiProvider || 'openai';
    if (aiKeyEl) aiKeyEl.value = s.aiApiKey || '';
    if (aiModelEl) aiModelEl.value = s.aiModel || '';
}

/**
 * حفظ إعدادات النظام
 * @param {Event} event - حدث النموذج
 */
function saveSystemSettings(event) {
    event.preventDefault();

    // الحصول على قيم الحقول
    const pointsPerResult = parseInt(document.getElementById('pointsPerResult').value) || 1;
    const defaultPointsForNewUser = parseInt(document.getElementById('defaultPointsForNewUser').value) || 50;

    // تحديث الإعدادات
    systemSettings = {
        pointsPerResult,
        defaultPointsForNewUser
    };

    // حفظ الإعدادات في localStorage
    localStorage.setItem('systemSettings', JSON.stringify(systemSettings));

    // عرض رسالة نجاح
    alert('تم حفظ إعدادات النظام بنجاح');
}

function saveApiSettings(event) {
    event.preventDefault();
    const savedSettings = localStorage.getItem('systemSettings');
    const s = savedSettings ? JSON.parse(savedSettings) : {};
    s.googleMapsApiKey = document.getElementById('systemGoogleMapsApiKey').value || '';
    s.aiProvider = document.getElementById('systemAiProvider').value || 'openai';
    s.aiApiKey = document.getElementById('systemAiApiKey').value || '';
    s.aiModel = document.getElementById('systemAiModel').value || '';
    localStorage.setItem('systemSettings', JSON.stringify(s));
    alert('تم حفظ إعدادات API بنجاح');
}

/**
 * تحميل إحصائيات النظام
 */
function loadStatistics() {
    // الحصول على قائمة المستخدمين
    const users = auth.getUsers();

    // عدد المستخدمين
    document.getElementById('totalUsersCount').textContent = users.length;

    // عدد المستخدمين النشطين
    const activeUsers = users.filter(user => user.isActive !== false);
    document.getElementById('activeUsersCount').textContent = activeUsers.length;

    // إجمالي عمليات البحث (مجموع عمليات البحث المحفوظة لكل المستخدمين)
    let totalSearches = 0;
    users.forEach(user => {
        totalSearches += (user.savedSearches?.length || 0);
    });
    document.getElementById('totalSearchesCount').textContent = totalSearches;
}

// هذا التعليق موجود لإلغاء التعريف المكرر لـ loadPendingAccounts
// تم نقل جميع الوظائف المتعلقة بالحسابات المعلقة إلى التعريفات السابقة في الملف

/**
 * إضافة مستمعي الأحداث لأزرار التفعيل والرفض
 */
function addPendingAccountsButtonListeners() {
    // أزرار تفعيل الحساب
    document.querySelectorAll('.activate-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            activateUserAccount(userId);
        });
    });

    // أزرار رفض الحساب
    document.querySelectorAll('.reject-user-btn').forEach(button => {
        button.addEventListener('click', () => {
            const userId = button.getAttribute('data-user-id');
            rejectUserAccount(userId);
        });
    });
}

/**
 * تفعيل حساب مستخدم
 * @param {string} userId - معرف المستخدم
 */
function activateUserAccount(userId) {
    // الحصول على قائمة المستخدمين
    const users = auth.getUsers();

    // تحديث بيانات المستخدم
    const updatedUsers = users.map(user => {
        if (user.id === userId) {
            return {
                ...user,
                isActive: true,
                activationPending: false,
                points: systemSettings.defaultPointsForNewUser || 50
            };
        }
        return user;
    });

    // حفظ التغييرات
    auth.saveUsers(updatedUsers);

    // إعادة تحميل بيانات المستخدمين
    loadPendingAccounts();
    loadUsers();

    // عرض رسالة نجاح
    alert('تم تفعيل الحساب بنجاح');
}

/**
 * رفض حساب مستخدم
 * @param {string} userId - معرف المستخدم
 */
function rejectUserAccount(userId) {
    // الحصول على قائمة المستخدمين
    const users = auth.getUsers();

    // حذف المستخدم من قائمة المستخدمين
    const updatedUsers = users.filter(user => user.id !== userId);

    // حفظ التغييرات
    auth.saveUsers(updatedUsers);

    // إعادة تحميل بيانات المستخدمين
    loadPendingAccounts();
    loadUsers();

    // عرض رسالة نجاح
    alert('تم رفض الحساب بنجاح');
}

/**
 * إنشاء حساب جديد من لوحة تحكم المدير
 */
function createNewUser() {
    console.log('محاولة إنشاء حساب جديد...');

    // الحصول على قيم الحقول
    const name = document.getElementById('createUserName').value.trim();
    const email = document.getElementById('createUserEmail').value.trim();
    const password = document.getElementById('createUserPassword').value.trim();
    const points = parseInt(document.getElementById('createUserPoints').value) || systemSettings.defaultPointsForNewUser;
    const isAdmin = document.getElementById('createUserIsAdmin').checked;
    const isActive = document.getElementById('createUserIsActive').checked;

    // عنصر عرض الخطأ
    const errorElement = document.getElementById('createUserError');

    // التحقق من البيانات
    if (!name || !email || !password) {
        errorElement.textContent = 'يجب تعبئة جميع الحقول المطلوبة';
        errorElement.classList.remove('d-none');
        return;
    }

    // التحقق من تنسيق البريد الإلكتروني
    const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailPattern.test(email)) {
        errorElement.textContent = 'البريد الإلكتروني غير صالح';
        errorElement.classList.remove('d-none');
        return;
    }

    // التحقق من أن البريد الإلكتروني غير مستخدم بالفعل
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    const emailExists = users.some(user => user.email.toLowerCase() === email.toLowerCase());

    if (emailExists) {
        errorElement.textContent = 'هذا البريد الإلكتروني مسجل مسبقًا';
        errorElement.classList.remove('d-none');
        return;
    }

    // تشفير كلمة المرور بنفس الطريقة التي يستخدمها نظام التسجيل العادي
    const hashedPassword = btoa(password); // تشفير بسيط باستخدام base64
    console.log('تم تشفير كلمة المرور للمستخدم الجديد');

    // إنشاء كائن المستخدم الجديد
    const newUser = {
        id: 'user_' + Date.now(),
        name: name,
        email: email,
        password: hashedPassword, // كلمة المرور مشفرة الآن
        points: points,
        isAdmin: isAdmin,
        isActive: isActive,
        activationPending: !isActive,
        registrationDate: new Date().toISOString(), // هذا الحقل يتناسق مع بقية النظام
        createdBy: 'admin',
        savedSearches: [] // تصحيح اسم الحقل ليتناسق مع بقية النظام
    };

    // حفظ المستخدم الجديد
    users.push(newUser);
    localStorage.setItem('users', JSON.stringify(users));

    // إغلاق النافذة المنبثقة
    const createUserModal = bootstrap.Modal.getInstance(document.getElementById('createUserModal'));
    createUserModal.hide();

    // إظهار رسالة نجاح
    let message;
    if (isActive) {
        message = `تم إنشاء حساب ${name} بنجاح وتفعيله`;
    } else {
        message = `تم إنشاء حساب ${name} بنجاح وهو بحاجة للتفعيل`;
    }
    showSuccessMessage(message);

    // تحديث قوائم المستخدمين
    loadUsers();
    loadPendingAccounts();

    // إعادة تعيين نموذج إنشاء الحساب
    document.getElementById('createUserForm').reset();
    errorElement.classList.add('d-none');
}

// تصدير الدوال للاستخدام في ملفات أخرى
window.adminPanel = {
    loadUsers,
    loadPendingAccounts,
    toggleUserStatus,
    activateUserAccount,
    rejectUserAccount,
    addPointsToUser,
    createNewUser,
    getSystemSettings: () => systemSettings
};
