'use client';

import { useState } from 'react';
import { X, ZoomIn } from 'lucide-react';

interface GallerySectionProps {
    images: string[];
    title?: string;
    description?: string;
    primaryColor?: string;
}

export default function GallerySection({
    images,
    title = "معرض الصور",
    description = "نظرة على أحدث مشاريعنا وأعمالنا المتميزة",
    primaryColor = "#16a34a"
}: GallerySectionProps) {
    const [selectedImage, setSelectedImage] = useState<string | null>(null);

    // If no images provided, don't render
    if (!images || images.length === 0 || images.every(img => !img)) return null;

    // Filter out empty strings
    const validImages = images.filter(img => img);

    return (
        <section className="py-20 bg-white">
            <div className="container mx-auto px-4">
                <div className="text-center mb-16">
                    <h2 className="text-3xl md:text-4xl font-bold mb-4" style={{ color: primaryColor }}>{title}</h2>
                    <p className="text-gray-600 max-w-2xl mx-auto">{description}</p>
                    <div className="w-24 h-1 mx-auto mt-6 rounded-full" style={{ backgroundColor: primaryColor }}></div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {validImages.map((img, index) => (
                        <div
                            key={index}
                            className="group relative h-64 md:h-80 rounded-2xl overflow-hidden cursor-pointer shadow-lg hover:shadow-2xl transition-all duration-300"
                            onClick={() => setSelectedImage(img)}
                        >
                            <img
                                src={img}
                                alt={`Gallery ${index + 1}`}
                                className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-110"
                            />
                            <div className="absolute inset-0 bg-black/0 group-hover:bg-black/30 transition-colors duration-300 flex items-center justify-center">
                                <ZoomIn className="text-white opacity-0 group-hover:opacity-100 transform scale-50 group-hover:scale-100 transition-all duration-300 w-10 h-10" />
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* Lightbox Modal */}
            {selectedImage && (
                <div
                    className="fixed inset-0 z-50 bg-black/90 backdrop-blur-sm flex items-center justify-center p-4 animate-in fade-in duration-200"
                    onClick={() => setSelectedImage(null)}
                >
                    <button
                        className="absolute top-6 right-6 text-white hover:text-gray-300 transition-colors bg-white/10 p-2 rounded-full"
                        onClick={() => setSelectedImage(null)}
                    >
                        <X className="w-8 h-8" />
                    </button>
                    <img
                        src={selectedImage}
                        alt="Full view"
                        className="max-w-full max-h-[90vh] rounded-lg shadow-2xl object-contain animate-in zoom-in-95 duration-300"
                        onClick={(e) => e.stopPropagation()}
                    />
                </div>
            )}
        </section>
    );
}
