import { Phone, MessageCircle, DollarSign, Armchair, RefreshCw, Truck, Clock, ThumbsUp, Wallet } from 'lucide-react';
import GallerySection from './shared/GallerySection';
import TestimonialsSection from './shared/TestimonialsSection';

export default function UsedFurnitureTemplate({ data }: { data: Record<string, string> }) {
    const primaryColor = data.primaryColor || '#d97706'; // Amber-600

    // Gallery Images
    const galleryImages = [
        data.galleryImage1,
        data.galleryImage2,
        data.galleryImage3,
        data.galleryImage4,
    ].filter(Boolean);

    // Testimonials
    const testimonials = [
        { name: data.clientName1, text: data.clientReview1, rating: 5 },
    ].filter(t => t.name && t.text);

    return (
        <div className="font-sans text-gray-800 bg-white min-h-full" dir="rtl">
            {/* Hero Section */}
            <header className="relative py-20 lg:py-32 overflow-hidden bg-amber-50">
                {/* Decorative Background Elements */}
                <div className="absolute top-0 right-0 w-64 h-64 bg-amber-200 rounded-full mix-blend-multiply filter blur-3xl opacity-30 animate-blob"></div>
                <div className="absolute top-0 left-0 w-64 h-64 bg-orange-200 rounded-full mix-blend-multiply filter blur-3xl opacity-30 animate-blob animation-delay-2000"></div>
                <div className="absolute -bottom-32 left-20 w-64 h-64 bg-yellow-200 rounded-full mix-blend-multiply filter blur-3xl opacity-30 animate-blob animation-delay-4000"></div>

                <div className="container mx-auto px-4 relative z-10">
                    <div className="grid lg:grid-cols-2 gap-12 items-center">
                        <div className="text-center lg:text-right">
                            <div className="inline-flex items-center gap-2 bg-white border border-amber-200 rounded-full px-4 py-1.5 text-amber-700 font-bold text-sm mb-8 shadow-sm animate-fade-in">
                                <Wallet className="w-4 h-4" />
                                <span>نشتري بأعلى الأسعار في السوق</span>
                            </div>
                            <h1 className="text-4xl lg:text-7xl font-black mb-6 text-gray-900 leading-tight">
                                {data.heroTitle} <br />
                                <span className="text-amber-600">والدفع فوري</span>
                            </h1>
                            <p className="text-xl text-gray-600 mb-10 leading-relaxed">{data.heroDescription}</p>

                            <div className="flex flex-col sm:flex-row gap-4 justify-center lg:justify-start">
                                <a href={`tel:${data.phone}`} className="bg-amber-600 hover:bg-amber-700 text-white px-8 py-4 rounded-xl font-bold flex items-center justify-center gap-3 transition-all shadow-lg hover:shadow-xl shadow-amber-200 transform hover:-translate-y-1">
                                    <Phone className="w-5 h-5" />
                                    اتصل بنا: {data.phone}
                                </a>
                                <a href={`https://wa.me/${data.whatsapp}`} className="bg-white text-gray-900 border-2 border-gray-100 hover:border-green-500 hover:text-green-600 px-8 py-4 rounded-xl font-bold flex items-center justify-center gap-3 transition-all transform hover:-translate-y-1">
                                    <MessageCircle className="w-5 h-5" />
                                    واتساب
                                </a>
                            </div>
                        </div>

                        <div className="relative animate-fade-in-up delay-200">
                            <div className="relative rounded-3xl overflow-hidden shadow-2xl border-4 border-white">
                                <img src="https://images.unsplash.com/photo-1556228453-efd6c1ff04f6?auto=format&fit=crop&q=80" alt="Used Furniture" className="w-full h-full object-cover" />
                                <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent flex items-end p-8">
                                    <div className="text-white">
                                        <p className="font-bold text-lg">خدمة سريعة في</p>
                                        <p className="text-amber-300 font-black text-2xl">{data.area}</p>
                                    </div>
                                </div>
                            </div>
                            {/* Floating Stats Card */}
                            <div className="absolute -bottom-6 -right-6 bg-white p-6 rounded-2xl shadow-xl border border-gray-100 hidden md:block">
                                <div className="flex items-center gap-4">
                                    <div className="w-12 h-12 bg-amber-100 rounded-full flex items-center justify-center">
                                        <ThumbsUp className="w-6 h-6 text-amber-600" />
                                    </div>
                                    <div>
                                        <p className="text-3xl font-black text-gray-900">{data.yearsExperience}+</p>
                                        <p className="text-sm text-gray-500 font-bold">سنوات خبرة</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </header>

            {/* Features (Why Choose Us) */}
            <section className="py-20 bg-white">
                <div className="container mx-auto px-4">
                    <div className="grid md:grid-cols-3 gap-8">
                        <div className="bg-gray-50 p-8 rounded-2xl border border-gray-100 hover:border-amber-200 transition-colors group">
                            <div className="w-14 h-14 bg-white rounded-xl shadow-sm flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <DollarSign className="w-7 h-7 text-amber-600" />
                            </div>
                            <h3 className="text-xl font-bold mb-3">{data.feature1 || 'أفضل الأسعار'}</h3>
                            <p className="text-gray-500">نقيم عفشك بما يرضي الله ونعطيك حقه وزيادة.</p>
                        </div>
                        <div className="bg-gray-50 p-8 rounded-2xl border border-gray-100 hover:border-amber-200 transition-colors group">
                            <div className="w-14 h-14 bg-white rounded-xl shadow-sm flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <Truck className="w-7 h-7 text-amber-600" />
                            </div>
                            <h3 className="text-xl font-bold mb-3">{data.feature2 || 'نقل مجاني'}</h3>
                            <p className="text-gray-500">نحن نتكفل بجميع تكاليف الفك والنقل، لن تتعب نفسك.</p>
                        </div>
                        <div className="bg-gray-50 p-8 rounded-2xl border border-gray-100 hover:border-amber-200 transition-colors group">
                            <div className="w-14 h-14 bg-white rounded-xl shadow-sm flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <Clock className="w-7 h-7 text-amber-600" />
                            </div>
                            <h3 className="text-xl font-bold mb-3">{data.feature3 || 'دفع فوري'}</h3>
                            <p className="text-gray-500">تستلم فلوسك كاش بيدك قبل ما نحرك العفش.</p>
                        </div>
                    </div>
                </div>
            </section>

            {/* What We Buy - Visual Grid */}
            <section className="py-20 bg-gray-900 text-white">
                <div className="container mx-auto px-4">
                    <div className="text-center mb-16">
                        <h2 className="text-3xl font-bold mb-4">ماذا نشتري؟</h2>
                        <div className="w-16 h-1 bg-amber-500 mx-auto rounded-full"></div>
                    </div>

                    <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
                        {['غرف نوم', 'مطابخ', 'مكيفات', 'أجهزة كهربائية', 'مجالس', 'سجاد', 'أثاث مكتبي', 'معدات مطاعم'].map((item, i) => (
                            <div key={i} className="bg-gray-800 p-6 rounded-xl text-center hover:bg-gray-700 transition-all cursor-default border border-gray-700 hover:border-amber-500/50 group">
                                <Armchair className="w-8 h-8 mx-auto mb-4 text-amber-500 group-hover:scale-110 transition-transform" />
                                <span className="font-bold text-lg">{item}</span>
                            </div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Gallery Section */}
            <GallerySection
                images={galleryImages}
                title={data.galleryTitle}
                primaryColor={primaryColor}
            />

            {/* Testimonials Section */}
            <TestimonialsSection
                testimonials={testimonials}
                primaryColor={primaryColor}
            />

            {/* Footer */}
            <footer className="bg-white border-t border-gray-100 py-12">
                <div className="container mx-auto px-4 text-center">
                    <h2 className="text-2xl font-black text-gray-900 mb-6">{data.companyName}</h2>
                    <p className="text-gray-500 mb-8">نشتري الاثاث المستعمل بأفضل الأسعار في {data.area}</p>

                    <div className="flex justify-center gap-4">
                        <a href={`tel:${data.phone}`} className="w-12 h-12 bg-amber-50 text-amber-600 rounded-full flex items-center justify-center hover:bg-amber-600 hover:text-white transition-all">
                            <Phone className="w-5 h-5" />
                        </a>
                        <a href={`https://wa.me/${data.whatsapp}`} className="w-12 h-12 bg-green-50 text-green-600 rounded-full flex items-center justify-center hover:bg-green-600 hover:text-white transition-all">
                            <MessageCircle className="w-5 h-5" />
                        </a>
                    </div>

                    <div className="mt-8 pt-8 border-t border-gray-100 text-sm text-gray-400">
                        &copy; {new Date().getFullYear()} {data.companyName}
                    </div>
                </div>
            </footer>
        </div>
    );
}
