'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useSession, signIn, signOut } from "next-auth/react";
import { templates } from '@/lib/templates';
import { Layout, Plus, Home as HomeIcon, Settings, Layers, ExternalLink, Copy, Check, ArrowLeft } from 'lucide-react';
import { Project } from '@/lib/storage';

export default function Dashboard() {
  const [activeTab, setActiveTab] = useState<'home' | 'projects' | 'templates'>('home');
  const [projects, setProjects] = useState<Project[]>([]);
  const [copiedId, setCopiedId] = useState<string | null>(null);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const { data: session, status } = useSession();

  const NavItem = ({ tab, icon: Icon, label }: { tab: typeof activeTab, icon: any, label: string }) => (
    <button
      onClick={() => {
        setActiveTab(tab);
        setIsMobileMenuOpen(false);
      }}
      className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${activeTab === tab ? 'bg-blue-50 text-blue-600 font-bold' : 'text-gray-600 hover:bg-gray-50'}`}
    >
      <Icon className="w-5 h-5" />
      {label}
    </button>
  );

  useEffect(() => {
    if (status === 'authenticated') {
      fetch('/api/projects')
        .then(res => {
          if (!res.ok) throw new Error('Failed to fetch');
          return res.json();
        })
        .then(data => {
          if (Array.isArray(data)) {
            setProjects(data);
          } else {
            setProjects([]);
          }
        })
        .catch(err => {
          console.error(err);
          setProjects([]);
        });
    } else {
      setProjects([]);
    }
  }, [status]);

  const copyLink = (slug: string) => {
    const url = `${window.location.origin}/p/${slug}`;
    navigator.clipboard.writeText(url);
    setCopiedId(slug);
    setTimeout(() => setCopiedId(null), 2000);
  };



  return (
    <div className="flex h-screen bg-gray-50 font-sans" dir="rtl">
      {/* Mobile Header */}
      <div className="md:hidden fixed top-0 left-0 right-0 bg-white border-b border-gray-200 p-4 z-50 flex justify-between items-center">
        <div className="flex items-center gap-3">
          <div className="w-8 h-8 bg-blue-600 rounded-lg flex items-center justify-center text-white font-bold">L</div>
          <span className="font-bold text-xl text-gray-800">لاندنج بيج</span>
        </div>
        <button onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)} className="p-2 text-gray-600">
          {isMobileMenuOpen ? <ArrowLeft className="w-6 h-6" /> : <Layout className="w-6 h-6" />}
        </button>
      </div>

      {/* Sidebar */}
      <aside className={`
        fixed inset-y-0 right-0 w-64 bg-white border-l border-gray-200 flex flex-col z-40 transform transition-transform duration-300 ease-in-out md:translate-x-0 md:static
        ${isMobileMenuOpen ? 'translate-x-0' : 'translate-x-full'}
      `}>
        <div className="p-6 border-b border-gray-100 hidden md:block">
          <div className="flex items-center gap-3">
            <div className="w-8 h-8 bg-blue-600 rounded-lg flex items-center justify-center text-white font-bold">L</div>
            <span className="font-bold text-xl text-gray-800">لاندنج بيج</span>
          </div>
        </div>

        <nav className="flex-1 p-4 space-y-2 mt-16 md:mt-0">
          <NavItem tab="home" icon={HomeIcon} label="الرئيسية" />
          <NavItem tab="projects" icon={Layers} label="صفحاتي" />
          <NavItem tab="templates" icon={Layout} label="النماذج" />
        </nav>

        <div className="p-4 border-t border-gray-100">
          {session ? (
            <div className="flex items-center gap-3 mb-4">
              {session.user?.image && (
                <img src={session.user.image} alt={session.user.name || ''} className="w-8 h-8 rounded-full" />
              )}
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium text-gray-900 truncate">{session.user?.name}</p>
                <p className="text-xs text-gray-500 truncate">{session.user?.email}</p>
              </div>
            </div>
          ) : null}

          {session ? (
            <button
              onClick={() => signOut()}
              className="w-full flex items-center gap-3 px-4 py-3 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
            >
              <Settings className="w-5 h-5" />
              تسجيل خروج
            </button>
          ) : (
            <button
              onClick={() => signIn('google')}
              className="w-full flex items-center gap-3 px-4 py-3 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
            >
              <Settings className="w-5 h-5" />
              تسجيل دخول
            </button>
          )}
        </div>
      </aside>

      {/* Overlay for mobile sidebar */}
      {isMobileMenuOpen && (
        <div
          className="fixed inset-0 bg-black/20 z-30 md:hidden"
          onClick={() => setIsMobileMenuOpen(false)}
        />
      )}

      {/* Main Content */}
      <main className="flex-1 overflow-y-auto p-4 md:p-8 mt-16 md:mt-0">
        <header className="flex flex-col md:flex-row justify-between items-start md:items-center mb-8 gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">
              {activeTab === 'home' && 'لوحة التحكم'}
              {activeTab === 'projects' && 'صفحاتي'}
              {activeTab === 'templates' && 'مكتبة النماذج'}
            </h1>
            <p className="text-gray-500 mt-1 text-sm md:text-base">
              {session ? `مرحباً ${session.user?.name?.split(' ')[0]}، ابدأ بإنشاء صفحة هبوط احترافية.` : 'أهلاً بك، سجل دخولك لحفظ مشاريعك.'}
            </p>
          </div>
          <button
            onClick={() => setActiveTab('templates')}
            className="w-full md:w-auto bg-blue-600 hover:bg-blue-700 text-white px-6 py-2.5 rounded-lg font-bold flex items-center justify-center gap-2 transition-colors shadow-sm"
          >
            <Plus className="w-5 h-5" />
            إنشاء صفحة جديدة
          </button>
        </header>

        {/* Stats (Home Only) */}
        {activeTab === 'home' && (
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 md:gap-6 mb-8">
            <div className="bg-white p-6 rounded-xl border border-gray-100 shadow-sm">
              <div className="text-gray-500 mb-2">إجمالي الصفحات</div>
              <div className="text-3xl font-bold text-gray-900">{projects.length}</div>
            </div>
            <div className="bg-white p-6 rounded-xl border border-gray-100 shadow-sm">
              <div className="text-gray-500 mb-2">الزيارات</div>
              <div className="text-3xl font-bold text-gray-900">
                {projects.reduce((sum, p) => sum + (p.views || 0), 0)}
              </div>
            </div>
            <div className="bg-white p-6 rounded-xl border border-gray-100 shadow-sm">
              <div className="text-gray-500 mb-2">النماذج المتاحة</div>
              <div className="text-3xl font-bold text-gray-900">{templates.length}</div>
            </div>
          </div>
        )}

        {/* Projects List */}
        {(activeTab === 'home' || activeTab === 'projects') && (
          <div className="mb-12">
            <h2 className="text-lg font-bold text-gray-800 mb-4 flex items-center justify-between">
              <span>آخر المشاريع</span>
              {activeTab === 'home' && (
                <button onClick={() => setActiveTab('projects')} className="text-sm text-blue-600 hover:underline">عرض الكل</button>
              )}
            </h2>

            {projects.length === 0 ? (
              <div className="bg-white rounded-xl border border-gray-200 p-12 text-center">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4 text-gray-400">
                  <Layers className="w-8 h-8" />
                </div>
                <h3 className="text-lg font-bold text-gray-900 mb-2">لا توجد صفحات بعد</h3>
                <p className="text-gray-500 mb-6">أنشئ صفحتك الأولى الآن وابدأ في استقبال العملاء.</p>
                <button
                  onClick={() => setActiveTab('templates')}
                  className="text-blue-600 font-bold hover:underline"
                >
                  تصفح النماذج
                </button>
              </div>
            ) : (
              <div className="bg-white rounded-xl border border-gray-200 overflow-hidden shadow-sm">
                <div className="overflow-x-auto">
                  <table className="w-full text-right min-w-[600px]">
                    <thead className="bg-gray-50 border-b border-gray-200">
                      <tr>
                        <th className="px-6 py-4 text-sm font-semibold text-gray-600">اسم الصفحة</th>
                        <th className="px-6 py-4 text-sm font-semibold text-gray-600">النموذج</th>
                        <th className="px-6 py-4 text-sm font-semibold text-gray-600">تاريخ الإنشاء</th>
                        <th className="px-6 py-4 text-sm font-semibold text-gray-600">الرابط</th>
                        <th className="px-6 py-4 text-sm font-semibold text-gray-600">الإجراءات</th>
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                      {projects.map((project) => {
                        const template = templates.find(t => t.id === project.templateId);
                        return (
                          <tr key={project.id} className="hover:bg-gray-50 transition-colors">
                            <td className="px-6 py-4 font-medium text-gray-900">{project.name || 'بدون عنوان'}</td>
                            <td className="px-6 py-4 text-gray-600">{template?.name || project.templateId}</td>
                            <td className="px-6 py-4 text-gray-500 text-sm">
                              {new Date(project.createdAt).toLocaleDateString('ar-SA')}
                            </td>
                            <td className="px-6 py-4">
                              <div className="flex items-center gap-2">
                                <a
                                  href={`/p/${project.slug}`}
                                  target="_blank"
                                  className="text-blue-600 hover:underline text-sm flex items-center gap-1"
                                >
                                  عرض <ExternalLink className="w-3 h-3" />
                                </a>
                                <button
                                  onClick={() => copyLink(project.slug)}
                                  className="text-gray-400 hover:text-gray-600 p-1 rounded"
                                  title="نسخ الرابط"
                                >
                                  {copiedId === project.slug ? <Check className="w-4 h-4 text-green-500" /> : <Copy className="w-4 h-4" />}
                                </button>
                              </div>
                            </td>
                            <td className="px-6 py-4">
                              <Link
                                href={`/builder/${project.templateId}?id=${project.id}`}
                                className="text-gray-600 hover:text-blue-600 font-medium text-sm"
                              >
                                تعديل
                              </Link>
                            </td>
                          </tr>
                        );
                      })}
                    </tbody>
                  </table>
                </div>
              </div>
            )}
          </div>
        )}

        {/* Templates Grid */}
        {(activeTab === 'templates') && (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {templates.map((template) => (
              <div key={template.id} className="bg-white rounded-xl border border-gray-200 overflow-hidden hover:shadow-md transition-shadow flex flex-col">
                <div className="h-48 bg-gray-100 relative border-b border-gray-100">
                  {/* Thumbnail placeholder */}
                  <div className="absolute inset-0 flex items-center justify-center text-gray-400">
                    <Layout className="w-12 h-12" />
                  </div>
                </div>
                <div className="p-5 flex flex-col flex-grow">
                  <h3 className="text-lg font-bold text-gray-900 mb-2">{template.name}</h3>
                  <p className="text-gray-500 text-sm mb-6 line-clamp-2 flex-grow">{template.description}</p>
                  <Link
                    href={`/builder/${template.id}`}
                    className="block w-full text-center bg-blue-600 hover:bg-blue-700 text-white font-medium py-2.5 rounded-lg transition-colors"
                  >
                    استخدم هذا النموذج
                  </Link>
                </div>
              </div>
            ))}
          </div>
        )}
      </main>
    </div>
  );
}
