import { NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { auth } from '@/auth';
import { nanoid } from 'nanoid';

export async function GET() {
    const session = await auth();

    if (!session?.user?.id) {
        // Allow anonymous users to see nothing (or maybe public projects in future)
        // Returning empty array prevents client-side crashes if it expects an array
        return NextResponse.json([]);
    }

    const projects = await prisma.project.findMany({
        where: {
            userId: session.user.id
        },
        orderBy: {
            createdAt: 'desc'
        }
    });

    // Parse data JSON string
    const parsedProjects = projects.map(p => ({
        ...p,
        data: JSON.parse(p.data)
    }));

    return NextResponse.json(parsedProjects);
}

export async function POST(request: Request) {
    try {
        const session = await auth();

        // Allow anonymous publishing (userId will be null)
        // if (!session?.user?.id) {
        //     return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
        // }

        const body = await request.json();
        console.log('Received publish request:', body);

        const projectData = {
            id: body.id || nanoid(),
            templateId: body.templateId,
            name: body.name,
            slug: body.slug || nanoid(10),
            data: JSON.stringify(body.data),
            userId: session?.user?.id || null // Allow null if not logged in
        };

        const project = await prisma.project.upsert({
            where: { id: projectData.id },
            update: {
                templateId: projectData.templateId,
                name: projectData.name,
                slug: projectData.slug,
                data: projectData.data,
                // Only update userId if we have a session, otherwise keep existing or null
                ...(session?.user?.id ? { userId: session.user.id } : {})
            },
            create: projectData
        });

        return NextResponse.json({
            ...project,
            data: JSON.parse(project.data)
        });
    } catch (error) {
        console.error('Error in POST /api/projects:', error);
        return NextResponse.json({ error: 'Failed to save project' }, { status: 500 });
    }
}
