<?php
/**
 * دوال مساعدة عامة
 * General Helper Functions
 */

/**
 * تنظيف المدخلات
 * Sanitize input
 */
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * التحقق من البريد الإلكتروني
 * Validate email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * توليد رمز عشوائي
 * Generate random token
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * توليد slug من النص
 * Generate slug from text
 */
function generateSlug($text) {
    // تحويل العربية إلى transliteration بسيط أو استخدام timestamp
    $slug = preg_replace('/[^a-z0-9-]/', '', strtolower(str_replace(' ', '-', $text)));
    
    if (empty($slug)) {
        $slug = 'page-' . time() . '-' . rand(1000, 9999);
    }
    
    return $slug;
}

/**
 * التحقق من وجود slug
 * Check if slug exists
 */
function slugExists($slug, $excludeId = null) {
    $sql = "SELECT id FROM projects WHERE slug = :slug";
    $params = ['slug' => $slug];
    
    if ($excludeId) {
        $sql .= " AND id != :id";
        $params['id'] = $excludeId;
    }
    
    $result = fetchOne($sql, $params);
    return !empty($result);
}

/**
 * توليد slug فريد
 * Generate unique slug
 */
function generateUniqueSlug($text, $excludeId = null) {
    $slug = generateSlug($text);
    $originalSlug = $slug;
    $counter = 1;
    
    while (slugExists($slug, $excludeId)) {
        $slug = $originalSlug . '-' . $counter;
        $counter++;
    }
    
    return $slug;
}

/**
 * إعادة توجيه
 * Redirect
 */
function redirect($url, $permanent = false) {
    if ($permanent) {
        header("HTTP/1.1 301 Moved Permanently");
    }
    header("Location: " . $url);
    exit;
}

/**
 * إرجاع JSON
 * Return JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * رسالة نجاح
 * Success response
 */
function success($message, $data = []) {
    jsonResponse([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

/**
 * رسالة خطأ
 * Error response
 */
function error($message, $statusCode = 400) {
    jsonResponse([
        'success' => false,
        'error' => $message
    ], $statusCode);
}

/**
 * التحقق من CSRF Token
 * Verify CSRF token
 */
function verifyCsrfToken($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * توليد CSRF Token
 * Generate CSRF token
 */
function generateCsrfToken() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = generateToken();
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * رفع صورة
 * Upload image
 */
function uploadImage($file, $folder = 'projects') {
    if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'error' => 'خطأ في رفع الملف'];
    }
    
    // التحقق من الحجم
    if ($file['size'] > UPLOAD_MAX_SIZE) {
        return ['success' => false, 'error' => 'حجم الملف كبير جداً'];
    }
    
    // التحقق من النوع
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ALLOWED_IMAGES)) {
        return ['success' => false, 'error' => 'نوع الملف غير مسموح'];
    }
    
    // إنشاء المجلد إذا لم يكن موجوداً
    $uploadDir = UPLOAD_PATH . '/' . $folder;
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // توليد اسم فريد
    $filename = uniqid() . '-' . time() . '.' . $ext;
    $destination = $uploadDir . '/' . $filename;
    
    // نقل الملف
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        $url = str_replace(APP_PATH, SITE_URL, $destination);
        return ['success' => true, 'url' => $url, 'filename' => $filename];
    }
    
    return ['success' => false, 'error' => 'فشل حفظ الملف'];
}

/**
 * تنسيق التاريخ بالعربية
 * Format date in Arabic
 */
function formatDate($date, $format = 'Y-m-d H:i:s') {
    if (is_string($date)) {
        $date = strtotime($date);
    }
    return date($format, $date);
}

/**
 * تحويل التاريخ للغة العربية
 * Convert date to Arabic
 */
function arabicDate($date) {
    $timestamp = is_string($date) ? strtotime($date) : $date;
    $day = date('d', $timestamp);
    $month = date('m', $timestamp);
    $year = date('Y', $timestamp);
    
    $arabicMonths = [
        '01' => 'يناير', '02' => 'فبراير', '03' => 'مارس',
        '04' => 'أبريل', '05' => 'مايو', '06' => 'يونيو',
        '07' => 'يوليو', '08' => 'أغسطس', '09' => 'سبتمبر',
        '10' => 'أكتوبر', '11' => 'نوفمبر', '12' => 'ديسمبر'
    ];
    
    return $day . ' ' . $arabicMonths[$month] . ' ' . $year;
}

/**
 * اختصار النص
 * Truncate text
 */
function truncate($text, $length = 100, $suffix = '...') {
    if (mb_strlen($text) > $length) {
        return mb_substr($text, 0, $length) . $suffix;
    }
    return $text;
}

/**
 * الحصول على عنوان IP الحقيقي
 * Get real IP address
 */
function getRealIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
