<?php
/**
 * API إدارة المشاريع
 * Projects Management API
 */

define('APP_PATH', dirname(__DIR__));
require_once APP_PATH . '/config/config.php';
require_once APP_PATH . '/config/database.php';
require_once APP_PATH . '/includes/functions.php';
require_once APP_PATH . '/auth/session.php';

header('Content-Type: application/json; charset=utf-8');

$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {
    case 'list':
        listProjects();
        break;
    
    case 'get':
        getProject();
        break;
    
    case 'create':
        createProject();
        break;
    
    case 'update':
        updateProject();
        break;
    
    case 'delete':
        deleteProject();
        break;
    
    default:
        error('إجراء غير صحيح');
}

/**
 * قائمة المشاريع
 */
function listProjects() {
    $user = getCurrentUser();
    
    if ($user) {
        // المشاريع الخاصة بالمستخدم المسجل
        $sql = "SELECT id, template_id, name, slug, views, created_at, updated_at 
                FROM projects 
                WHERE user_id = :user_id 
                ORDER BY created_at DESC";
        $projects = fetchAll($sql, ['user_id' => $user['id']]);
    } else {
        $projects = [];
    }
    
    success('تم جلب المشاريع بنجاح', $projects);
}

/**
 * جلب مشروع محدد
 */
function getProject() {
    $id = $_GET['id'] ?? null;
    $slug = $_GET['slug'] ?? null;
    
    if (!$id && !$slug) {
        error('يجب تحديد ID أو Slug');
    }
    
    if ($id) {
        $sql = "SELECT * FROM projects WHERE id = :id";
        $params = ['id' => $id];
    } else {
        $sql = "SELECT * FROM projects WHERE slug = :slug";
        $params = ['slug' => $slug];
    }
    
    $project = fetchOne($sql, $params);
    
    if (!$project) {
        error('المشروع غير موجود', 404);
    }
    
    // فك تشفير البيانات JSON
    $project['data'] = json_decode($project['data'], true);
    
    success('تم جلب المشروع بنجاح', $project);
}

/**
 * إنشاء مشروع جديد
 */
function createProject() {
    requireLogin();
    
    $user = getCurrentUser();
    $templateId = $_POST['template_id'] ?? '';
    $name = $_POST['name'] ?? '';
    $data = $_POST['data'] ?? '';
    
    // التحقق من البيانات
    if (empty($templateId) || empty($name)) {
        error('يرجى تعبئة جميع الحقول المطلوبة');
    }
    
    // التحقق من صحة JSON
    $formData = json_decode($data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error('بيانات غير صحيحة');
    }
    
    // توليد slug فريد
    $slug = generateUniqueSlug($name);
    
    // إدراج المشروع
    $projectData = [
        'user_id' => $user['id'],
        'template_id' => sanitize($templateId),
        'name' => sanitize($name),
        'slug' => $slug,
        'data' => $data,
        'views' => 0
    ];
    
    $projectId = insert('projects', $projectData);
    
    if ($projectId) {
        success('تم إنشاء المشروع بنجاح', [
            'id' => $projectId,
            'slug' => $slug
        ]);
    } else {
        error('فشل إنشاء المشروع');
    }
}

/**
 * تحديث مشروع
 */
function updateProject() {
    requireLogin();
    
    $user = getCurrentUser();
    $id = $_POST['id'] ?? '';
    $name = $_POST['name'] ?? '';
    $data = $_POST['data'] ?? '';
    
    if (empty($id)) {
        error('يجب تحديد ID المشروع');
    }
    
    // التحقق من ملكية المشروع
    $project = fetchOne("SELECT * FROM projects WHERE id = :id AND user_id = :user_id", [
        'id' => $id,
        'user_id' => $user['id']
    ]);
    
    if (!$project) {
        error('المشروع غير موجود أو ليس لديك صلاحية', 403);
    }
    
    // التحقق من صحة JSON
    if (!empty($data)) {
        $formData = json_decode($data, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error('بيانات غير صحيحة');
        }
    }
    
    // تحضير البيانات للتحديث
    $updateData = [];
    
    if (!empty($name)) {
        $updateData['name'] = sanitize($name);
        // تحديث slug إذا تغير الاسم
        if ($name !== $project['name']) {
            $updateData['slug'] = generateUniqueSlug($name, $id);
        }
    }
    
    if (!empty($data)) {
        $updateData['data'] = $data;
    }
    
    if (empty($updateData)) {
        error('لا توجد بيانات للتحديث');
    }
    
    // تحديث المشروع
    $result = update('projects', $updateData, 'id = :id', ['id' => $id]);
    
    if ($result) {
        success('تم تحديث المشروع بنجاح', [
            'id' => $id,
            'slug' => $updateData['slug'] ?? $project['slug']
        ]);
    } else {
        error('فشل تحديث المشروع');
    }
}

/**
 * حذف مشروع
 */
function deleteProject() {
    requireLogin();
    
    $user = getCurrentUser();
    $id = $_POST['id'] ?? $_GET['id'] ?? '';
    
    if (empty($id)) {
        error('يجب تحديد ID المشروع');
    }
    
    // التحقق من ملكية المشروع
    $project = fetchOne("SELECT * FROM projects WHERE id = :id AND user_id = :user_id", [
        'id' => $id,
        'user_id' => $user['id']
    ]);
    
    if (!$project) {
        error('المشروع غير موجود أو ليس لديك صلاحية', 403);
    }
    
    // حذف المشروع
    $result = delete('projects', 'id = :id', ['id' => $id]);
    
    if ($result) {
        success('تم حذف المشروع بنجاح');
    } else {
        error('فشل حذف المشروع');
    }
}
